// -*- Mode: Go; indent-tabs-mode: t -*-

/*
 * Copyright (C) 2024 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

package builtin_test

import (
	. "gopkg.in/check.v1"

	"github.com/snapcore/snapd/interfaces"
	"github.com/snapcore/snapd/interfaces/apparmor"
	"github.com/snapcore/snapd/interfaces/builtin"
	"github.com/snapcore/snapd/snap"
	"github.com/snapcore/snapd/testutil"
)

type SnapPromptingControlInterfaceSuite struct {
	iface    interfaces.Interface
	slotInfo *snap.SlotInfo
	slot     *interfaces.ConnectedSlot
	plugInfo *snap.PlugInfo
	plug     *interfaces.ConnectedPlug
}

var _ = Suite(&SnapPromptingControlInterfaceSuite{
	iface: builtin.MustInterface("snap-interfaces-requests-control"),
})

func (s *SnapPromptingControlInterfaceSuite) SetUpTest(c *C) {
	const coreSlotYaml = `
name: core
type: os
version: 1.0
slots:
  snap-interfaces-requests-control:
`
	s.slot, s.slotInfo = MockConnectedSlot(c, coreSlotYaml, nil, "snap-interfaces-requests-control")

	const appPlugYaml = `
name: other
version: 0
apps:
  app:
    command: foo
    plugs: [snap-interfaces-requests-control]
`
	s.plug, s.plugInfo = MockConnectedPlug(c, appPlugYaml, nil, "snap-interfaces-requests-control")
}

func (s *SnapPromptingControlInterfaceSuite) TestName(c *C) {
	c.Check(s.iface.Name(), Equals, "snap-interfaces-requests-control")
}

func (s *SnapPromptingControlInterfaceSuite) TestSanitizeSlot(c *C) {
	c.Check(interfaces.BeforePrepareSlot(s.iface, s.slotInfo), IsNil)
}

func (s *SnapPromptingControlInterfaceSuite) TestSanitizePlugTrivial(c *C) {
	c.Check(interfaces.BeforePreparePlug(s.iface, s.plugInfo), IsNil)
}

func (s *SnapPromptingControlInterfaceSuite) TestSanitizePlugWithUserHandler(c *C) {
	const appPlugYaml = `
name: other
version: 0
apps:
  app:
    command: foo
    daemon: simple
    daemon-scope: user

plugs:
  snap-interfaces-requests-control:
    handler-service: app
`
	_, plugInfo := MockConnectedPlug(c, appPlugYaml, nil, "snap-interfaces-requests-control")
	c.Check(interfaces.BeforePreparePlug(s.iface, plugInfo), IsNil)
}

func (s *SnapPromptingControlInterfaceSuite) TestSanitizePlugWBadHandlerNotDaemon(c *C) {
	const appPlugYaml = `
name: other
version: 0
apps:
  app:
    command: foo

plugs:
  snap-interfaces-requests-control:
    handler-service: app
`
	_, plugInfo := MockConnectedPlug(c, appPlugYaml, nil, "snap-interfaces-requests-control")
	c.Check(interfaces.BeforePreparePlug(s.iface, plugInfo), ErrorMatches, `declared handler service \"app\" is not a user service`)
}

func (s *SnapPromptingControlInterfaceSuite) TestSanitizePlugWBadHandlerNotUserDaemon(c *C) {
	const appPlugYaml = `
name: other
version: 0
apps:
  app:
    command: foo
    daemon: simple

plugs:
  snap-interfaces-requests-control:
    handler-service: app
`
	_, plugInfo := MockConnectedPlug(c, appPlugYaml, nil, "snap-interfaces-requests-control")
	c.Check(interfaces.BeforePreparePlug(s.iface, plugInfo), ErrorMatches, `declared handler service \"app\" is not a user service`)
}

func (s *SnapPromptingControlInterfaceSuite) TestSanitizePlugWBadHandlerNotFound(c *C) {
	const appPlugYaml = `
name: other
version: 0
apps:
  app:
    command: foo

plugs:
  snap-interfaces-requests-control:
    handler-service: app-not-found
`
	_, plugInfo := MockConnectedPlug(c, appPlugYaml, nil, "snap-interfaces-requests-control")
	c.Check(interfaces.BeforePreparePlug(s.iface, plugInfo), ErrorMatches, `declared handler service \"app-not-found\" not found in snap`)
}

func (s *SnapPromptingControlInterfaceSuite) TestSanitizePlugWBadHandlerBadType(c *C) {
	const appPlugYaml = `
name: other
version: 0
apps:
  app:
    command: foo

plugs:
  snap-interfaces-requests-control:
    handler-service: 123.2
`
	_, plugInfo := MockConnectedPlug(c, appPlugYaml, nil, "snap-interfaces-requests-control")
	c.Check(interfaces.BeforePreparePlug(s.iface, plugInfo), ErrorMatches, `snap-interfaces-requests-control \"handler-service\" attribute must be a string, not float64 123.2`)
}

func (s *SnapPromptingControlInterfaceSuite) TestAppArmor(c *C) {
	// The interface generates no AppArmor rules
	appSet, err := interfaces.NewSnapAppSet(s.plug.Snap(), nil)
	c.Assert(err, IsNil)
	spec := apparmor.NewSpecification(appSet)
	c.Check(spec.SecurityTags(), HasLen, 0)
	c.Assert(spec.AddConnectedPlug(s.iface, s.plug, s.slot), IsNil)
	c.Assert(spec.SecurityTags(), DeepEquals, []string{"snap.other.app"})
	c.Check(spec.SnippetForTag("snap.other.app"), testutil.Contains,
		"peer=(name=com.canonical.Shell.PermissionPrompting)")
	c.Check(spec.SnippetForTag("snap.other.app"), testutil.Contains,
		"path=/com/canonical/Shell/PermissionPrompting\n")
	c.Check(spec.SnippetForTag("snap.other.app"), testutil.Contains,
		"interface=com.canonical.Shell.PermissionPrompting\n")

	appSet, err = interfaces.NewSnapAppSet(s.plug.Snap(), nil)
	c.Assert(err, IsNil)
	spec = apparmor.NewSpecification(appSet)
	c.Assert(spec.AddConnectedSlot(s.iface, s.plug, s.slot), IsNil)
	c.Check(spec.SecurityTags(), HasLen, 0)

	appSet, err = interfaces.NewSnapAppSet(s.plug.Snap(), nil)
	c.Assert(err, IsNil)
	spec = apparmor.NewSpecification(appSet)
	c.Assert(spec.AddPermanentPlug(s.iface, s.plugInfo), IsNil)
	c.Check(spec.SecurityTags(), HasLen, 0)

	appSet, err = interfaces.NewSnapAppSet(s.plug.Snap(), nil)
	c.Assert(err, IsNil)
	spec = apparmor.NewSpecification(appSet)
	c.Assert(spec.AddPermanentSlot(s.iface, s.slotInfo), IsNil)
	c.Check(spec.SecurityTags(), HasLen, 0)
}

func (s *SnapPromptingControlInterfaceSuite) TestInterfaces(c *C) {
	c.Check(builtin.Interfaces(), testutil.DeepContains, s.iface)
}
