// Copyright (c) .NET Foundation. All rights reserved.
// Licensed under the Apache License, Version 2.0. See License.txt in the project root for license information.

#nullable enable

using System;
using System.Threading;
using System.Threading.Tasks;
using FluentAssertions;
using Moq;
using NuGet.Common;
using NuGet.Configuration;
using NuGet.Protocol;
using NuGet.Protocol.Core.Types;
using NuGet.Protocol.Model;
using Xunit;

namespace NuGet.Commands.Test.RestoreCommandTests
{
    public class VulnerabilityInformationProviderTests
    {
        [Fact]
        public async Task GetVulnerabilityInformationAsync_CalledMultipleTimes_CachesValueAsync()
        {
            // Arrange
            var vulnerabilityResource = new Mock<IVulnerabilityInfoResource>();
            vulnerabilityResource.Setup(x => x.GetVulnerabilityInfoAsync(It.IsAny<SourceCacheContext>(), It.IsAny<ILogger>(), It.IsAny<CancellationToken>()))
                .ReturnsAsync(new GetVulnerabilityInfoResult(knownVulnerabilities: null, exceptions: new System.AggregateException(new Exception("Test"))));

            var resourceProvider = new Mock<INuGetResourceProvider>();
            resourceProvider.Setup(x => x.TryCreate(It.IsAny<SourceRepository>(), It.IsAny<CancellationToken>()))
                .ReturnsAsync(new Tuple<bool, INuGetResource?>(true, vulnerabilityResource.Object));
            resourceProvider.SetupGet(x => x.ResourceType).Returns(typeof(IVulnerabilityInfoResource));

            var packageSource = new PackageSource("https://unit.test/v3/index.json");
            var source = new SourceRepository(packageSource, new[] { resourceProvider.Object });
            var logger = new Mock<ILogger>();
            var provider = new VulnerabilityInformationProvider(source, logger.Object, isAuditSource: false);

            // Act
            for (int i = 0; i < 10; i++)
            {
                var result = await provider.GetVulnerabilityInformationAsync(CancellationToken.None);
                Assert.NotNull(result);
            }

            // Assert
            vulnerabilityResource.Verify(x => x.GetVulnerabilityInfoAsync(It.IsAny<SourceCacheContext>(), It.IsAny<ILogger>(), It.IsAny<CancellationToken>()), Times.Once);
        }

        // While IVulnerabilityInfoResource.GetVulnerabilityInfoAsync is designed to never throw, the provider needs to get the resource from the service index
        // and that can throw. Normally it would throw when resolving the graph, but if all packages are already in the global packages folder, then the
        // SourceRepository instances haven't yet attempted to load the service index. This test ensures that the provider does not throw if the service index
        // cannot be retrieved, which might happen for reasons like the DNS lookup failed, the service index itself returns an HTTP 404 or 401, network is down, etc.
        [Fact]
        public async Task GetVulnerabilityInformationAsync_SourceWithInvalidHostname_DoesNotThrow()
        {
            // Arrange
            // .test is a reserved TLD, so we know it will never exist.
            var packageSource = new PackageSource("https://nuget.test/v3/index.json");
            SourceRepository source = Repository.Factory.GetCoreV3(packageSource);
            var logger = NullLogger.Instance;
            var provider = new VulnerabilityInformationProvider(source, logger, isAuditSource: false);

            // Act
            var result = await provider.GetVulnerabilityInformationAsync(CancellationToken.None);

            // Assert
            result.Should().NotBeNull();
            result!.KnownVulnerabilities.Should().BeNull();
            result.Exceptions.Should().NotBeNull();
        }
    }
}
