/*****************************************************************
 * gmerlin - a general purpose multimedia framework and applications
 *
 * Copyright (c) 2001 - 2012 Members of the Gmerlin project
 * gmerlin-general@lists.sourceforge.net
 * http://gmerlin.sourceforge.net
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * *****************************************************************/

#include <config.h>

#include <mediadb_private.h>
#include <gmerlin/log.h>
#include <gmerlin/utils.h>
#include <string.h>

#define LOG_DOMAIN "db.audioalbum"

#define ARTIST_COL        1
#define TITLE_COL         2
#define SEARCH_TITLE_COL  3
#define GENRE_COL         4
#define COVER_COL         5
#define DATE_COL          6

static int query_audioalbum(bg_db_t * db, void * a1)
  {
  int result;
  int found = 0;
  bg_db_audio_album_t * a = a1;
  sqlite3_stmt * st = db->q_audio_albums;
  sqlite3_bind_int64(st, 1, a->obj.id);

  if((result = sqlite3_step(st)) == SQLITE_ROW)
    {
    BG_DB_GET_COL_INT(ARTIST_COL, a->artist_id);
    BG_DB_GET_COL_STRING(TITLE_COL, a->title);
    BG_DB_GET_COL_STRING(SEARCH_TITLE_COL, a->search_title);
    BG_DB_GET_COL_INT(GENRE_COL, a->genre_id);
    BG_DB_GET_COL_INT(COVER_COL, a->cover_id);
    BG_DB_GET_COL_DATE(DATE_COL, a->date);
    found = 1;
    }
  sqlite3_reset(st);
  sqlite3_clear_bindings(st);
  
  if(!found)
    return 0;
  
  
  a->artist = bg_db_string_cache_get(db->audio_artists, db->db,
                                     a->artist_id);
  a->genre  = bg_db_string_cache_get(db->audio_genres, db->db,
                                     a->genre_id);
  return 1;
  }

static void get_children_audioalbum(bg_db_t * db, void * obj, bg_sqlite_id_tab_t * tab)
  {
  char * sql;
  sql = sqlite3_mprintf("SELECT ID FROM AUDIO_FILES WHERE ALBUM = %"PRId64" ORDER BY TRACK;",
                        bg_db_object_get_id(obj));
  bg_sqlite_exec(db->db, sql, bg_sqlite_append_id_callback, tab);
  sqlite3_free(sql);
  }

static void update_audioalbum(bg_db_t * db, void * obj)
  {
  bg_db_audio_album_t * a = (bg_db_audio_album_t *)obj;
  char * sql;
  char date_string[BG_DB_DATE_STRING_LEN];
  
  bg_db_date_to_string(&a->date, date_string);
  
  sql = sqlite3_mprintf("UPDATE AUDIO_ALBUMS SET DATE = %Q, GENRE = %"PRId64", COVER = %"PRId64" WHERE ID = %"PRId64";",
                        date_string, a->genre_id, a->cover_id, bg_db_object_get_id(a));
  bg_sqlite_exec(db->db, sql, NULL, NULL);
  sqlite3_free(sql);
  return;
  }


static void del_audioalbum(bg_db_t * db, bg_db_object_t * obj) // Delete from db
  {
  bg_db_audio_album_t*a = (bg_db_audio_album_t*)obj;
  /* If we had an album cover, we need to set it's type back to
     image file */
  if(a->cover_id > 0)
    {
    bg_db_object_t * cover = bg_db_object_query(db, a->cover_id);

    // Cover can be disappeared already if a whole directory vanished
    if(cover)
      {
      bg_db_object_set_type(cover, BG_DB_OBJECT_IMAGE_FILE);
      bg_db_object_unref(cover);
      }
    }

  bg_sqlite_delete_by_id(db->db, "AUDIO_ALBUMS", obj->id);
  }

static void free_audioalbum(void * obj)
  {
  bg_db_audio_album_t*a = obj;
  if(a->artist)
    free(a->artist);
  if(a->title)
    free(a->title);
  if(a->search_title)
    free(a->search_title);
  if(a->genre)
    free(a->genre);

  }

static void dump_audioalbum(void * obj)
  {
  bg_db_audio_album_t*a = obj;
  gavl_diprintf(2, "Artist:   %s\n", a->artist);
  gavl_diprintf(2, "Title:    %s (%s)\n", a->title, a->search_title);
  gavl_diprintf(2, "Year:     %d\n", a->date.year);
  gavl_diprintf(2, "Genre:    %s\n", a->genre);
  gavl_diprintf(2, "Cover ID: %"PRId64"\n", a->cover_id);
  }

const bg_db_object_class_t bg_db_audio_album_class =
  {
    .name = "Audio album",
    .del = del_audioalbum,
    .free = free_audioalbum,
    .query = query_audioalbum,
    .update = update_audioalbum,
    .dump = dump_audioalbum,
    .get_children = get_children_audioalbum,
    .parent = NULL, // Object
  };

static int compare_album_by_file(const bg_db_object_t * obj, const void * data)
  {
  const bg_db_audio_file_t * file;
  const bg_db_audio_album_t * album;

  if(obj->type == BG_DB_OBJECT_AUDIO_ALBUM)
    {
    file = (const bg_db_audio_file_t*)data;
    album = (const bg_db_audio_album_t*)obj;
    
    if(!strcmp(file->album, album->title) &&
       (album->artist_id == file->albumartist_id))
      return 1;
    }
  return 0;
  } 

  /* Check if the album already exists */

static int64_t find_by_file(bg_db_t * db, bg_db_audio_file_t * f)
  {
  int64_t ret;
  char * sql;
  int result;

  ret = bg_db_cache_search(db, compare_album_by_file, f);
  if(ret > 0)
    return ret;

  ret = -1;
  sql = sqlite3_mprintf("select ID from AUDIO_ALBUMS where ((TITLE = %Q) & (ARTIST = %"PRId64"));",
                        f->album, f->albumartist_id);
  result = bg_sqlite_exec(db->db, sql, bg_sqlite_int_callback, &ret);
  sqlite3_free(sql);

  if(!result)
    return -1; // Should never happen

  return ret;
  
  }

void bg_db_audio_file_add_to_album(bg_db_t * db, bg_db_audio_file_t * f)
  {
  int64_t id;
  bg_db_audio_album_t * a;
  if((id = find_by_file(db, f)) > 0)
    {
    int changed = 0;
    /* Add to album */
    a = bg_db_object_query(db, id);
    bg_db_object_add_child(db, a, f);

    /* Clear some fields if they changed */

    /* If there is a date set, we check if it's the same for the added track */
    if(!bg_db_date_equal(&a->date, &f->date))
      {
      bg_db_date_set_invalid(&a->date);
      changed = 1;
      }
    /* If there is a genre set, we check if it's the same for the added track */
    if(a->genre_id != f->genre_id)
      {
      a->genre_id = 0;
      changed = 1;
      }
    f->album_id = bg_db_object_get_id(a);

    if(changed)
      bg_db_create_vfolders(db, a);

    bg_db_object_unref(a);
    }
  else
    {
    char * label;
    char * sql;
    char date_string[BG_DB_DATE_STRING_LEN];
    
    /* Create album */
    a = bg_db_object_create(db);
    bg_db_object_set_type(a, BG_DB_OBJECT_AUDIO_ALBUM);
    bg_db_object_set_parent_id(db, a, -1);
    a->genre_id = f->genre_id;
    a->genre = gavl_strdup(f->genre);
    a->cover_id = -1;
    /* Remove title completely? */
    a->title = gavl_strdup(f->album);
    a->search_title = gavl_strdup(bg_db_get_search_string(a->title));
    
    gavl_log(GAVL_LOG_INFO, LOG_DOMAIN, "Creating album %s", a->title);

    // #if 0    
    if(f->albumartist)
      label = bg_sprintf("%s - %s", f->albumartist, f->album);
    else
      label = gavl_strdup(f->album);
    
    bg_db_object_set_label(a, label);
    free(label);
    
    bg_db_date_copy(&a->date, &f->date);
    a->artist_id = f->albumartist_id;
    a->artist = gavl_strdup(f->albumartist);
    
    gavl_log(GAVL_LOG_INFO, LOG_DOMAIN, "Adding audio album %s", a->title);
    bg_db_date_to_string(&a->date, date_string);
    
    sql = sqlite3_mprintf("INSERT INTO AUDIO_ALBUMS ( ID, ARTIST, TITLE, SEARCH_TITLE, GENRE, DATE ) VALUES ( %"PRId64", %"PRId64", %Q, %Q, %"PRId64", %Q);",
                          bg_db_object_get_id(a), a->artist_id, a->title,
                          a->search_title, a->genre_id, date_string);
    bg_sqlite_exec(db->db, sql, NULL, NULL);
    sqlite3_free(sql);

    f->album_id = bg_db_object_get_id(a);
    
    bg_db_object_add_child(db, a, f);
    bg_db_create_vfolders(db, a);
    bg_db_object_unref(a);
    }
  }

void bg_db_audio_file_remove_from_album(bg_db_t * db, bg_db_audio_file_t * f)
  {
  bg_db_audio_album_t * a;

  if(f->album_id <= 0) // Some files have no album
    return;
  
  a = bg_db_object_query(db, f->album_id);

  
  if(!a)
    return; // Shouldn't happen

  bg_db_object_remove_child(db, a, f);

  if(!a->obj.children)
    {
    gavl_log(GAVL_LOG_INFO, LOG_DOMAIN, "Removing empty album %s", a->title);
    bg_db_object_delete(db, a);
    }
  else
    bg_db_object_unref(a);
  
  }

