/*
 * Copyright (C) 2000-2025 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <strings.h>
#include <xine/sorted_array.h>

#include "dump.h"

#include "common.h"
#include "kbindings_common.h"
#include "kbindings.h"
#include "actions.h"
#include "xine-toolkit/cfg_parse.h"

#undef _KBT_DEBUG_REFS

/** refs string layout:
 *  always 4 (longword) aligned.
 *  [-4][-3] unused.
 *  [-2] num_longwords_with_padding, non default_binding_table only.
 *  [-1] reference counter, non default_binding_table only, 0 for "not refcounted".
 *  [0].. zero padded string: "whatever\0\0\0"
 *  [(strlen(string)+4)/4*4] safety longword to stop fast cmp loops. */
typedef union {
  int32_t i[32];
  char s[4 * 32];
} refs_dummy_t;

static ATTR_INLINE_ALL_STRINGOPS char *refs_set_dummy (refs_dummy_t *dummy, const char *text) {
  size_t n, l = strlen (text);
  if (l > sizeof (*dummy) - 9)
    l = sizeof (*dummy) - 9;
  n = l >> 2;
  dummy->i[1 + n] = 0;
  dummy->i[1 + n + 1] = ~0u;
  memcpy (dummy->s + 4, text, l + 1);
  dummy->i[0] = 0;
  dummy->s[2] = n + 1;
  return dummy->s + 4;
}

#define refs_str_void "\x00\x00\x02\x00VOID\x00\x00\x00\x00\x00\x00\x00"

/*
 * Default key mapping table.
 */
typedef struct {
  const char  *comment;                   /* human readable description. */
  union {char s[24]; int32_t v[6];} name; /* Human readable action, used in config file too. */
  union {char s[12]; int32_t v[3];} key;  /* key binding. */
  uint16_t    id;                         /* action_id_t. */
  uint8_t     modifier;                   /* modifier flags. */
  uint8_t     flags;                      /* is KBE_FLAG_*. */
} default_binding_t;

static const default_binding_t default_binding_table[] = {
  { N_("start playback"),
    {"Play"},                   {"Return"},     ACTID_PLAY,                      MODIFIER_NOMOD,    0},
  { N_("playback pause toggle"),
    {"Pause"},                  {"space"},      ACTID_PAUSE,                     MODIFIER_NOMOD,    0},
  { N_("stop playback"),
    {"Stop"},                   {"S"},          ACTID_STOP,                      MODIFIER_NOMOD,    0},
  { N_("close stream"),
    {"Close"},                  {"C"},          ACTID_CLOSE,                     MODIFIER_NOMOD,    0},
  { N_("take a snapshot"),
    {"Snapshot"},               {"t"},          ACTID_SNAPSHOT,                  MODIFIER_NOMOD,    0},
  { N_("eject the current medium"),
    {"Eject"},                  {"e"},          ACTID_EJECT,                     MODIFIER_NOMOD,    0},
  { N_("show MRL or ident"),
    {"ShowMrl"},                {"M"},          ACTID_SHOW_MRL,                  MODIFIER_CTRL,     KBE_FLAG_gui},
  { N_("show elapsed or remaining time"),
    {"ShowTime"},               {"T"},          ACTID_SHOW_TIME,                 MODIFIER_CTRL,     KBE_FLAG_gui},
  { N_("select and play next MRL in the playlist"),
    {"NextMrl"},                {"Next"},       ACTID_MRL_NEXT,                  MODIFIER_NOMOD,    0},
  { N_("select and play previous MRL in the playlist"),
    {"PriorMrl"},               {"Prior"},      ACTID_MRL_PRIOR,                 MODIFIER_NOMOD,    0},
  { N_("select and play MRL in the playlist"),
    {"SelectMrl"},              {"Select"},     ACTID_MRL_SELECT,                MODIFIER_NOMOD,    0},
  { N_("loop mode toggle"),
    {"ToggleLoopMode"},         {"l"},          ACTID_LOOPMODE,                  MODIFIER_NOMOD,    0},
  { N_("stop playback after played stream"),
    {"PlaylistStop"},           {"l"},          ACTID_PLAYLIST_STOP,             MODIFIER_CTRL,     0},
  { N_("scan playlist to grab stream infos"),
    {"ScanPlaylistInfo"},       {"s"},          ACTID_SCANPLAYLIST,              MODIFIER_CTRL,     0},
  { N_("add a mediamark from current playback"),
    {"AddMediamark"},           {"a"},          ACTID_ADDMEDIAMARK,              MODIFIER_CTRL,     0},
  { N_("edit selected mediamark"),
    {"MediamarkEditor"},        {"e"},          ACTID_MMKEDITOR,                 MODIFIER_CTRL,     KBE_FLAG_gui},
  { N_("set position to -60 seconds in current stream"),
    {"SeekRelative-60"},        {"Left"},       ACTID_SEEK_REL_m60,              MODIFIER_NOMOD,    0},
  { N_("set position to +60 seconds in current stream"),
    {"SeekRelative+60"},        {"Right"},      ACTID_SEEK_REL_p60,              MODIFIER_NOMOD,    0},
  { N_("set position to -30 seconds in current stream"),
    {"SeekRelative-30"},        {"Left"},       ACTID_SEEK_REL_m30,              MODIFIER_META,     0},
  { N_("set position to +30 seconds in current stream"),
    {"SeekRelative+30"},        {"Right"},      ACTID_SEEK_REL_p30,              MODIFIER_META,     0},
  { N_("set position to -15 seconds in current stream"),
    {"SeekRelative-15"},        {"Left"},       ACTID_SEEK_REL_m15,              MODIFIER_CTRL,     0},
  { N_("set position to +15 seconds in current stream"),
    {"SeekRelative+15"},        {"Right"},      ACTID_SEEK_REL_p15,              MODIFIER_CTRL,     0},
  { N_("set position to -7 seconds in current stream"),
    {"SeekRelative-7"},         {"Left"},       ACTID_SEEK_REL_m7,               MODIFIER_MOD3,     0},
  { N_("set position to +7 seconds in current stream"),
    {"SeekRelative+7"},         {"Right"},      ACTID_SEEK_REL_p7,               MODIFIER_MOD3,     0},
  { N_("set position to beginning of current stream"),
    {"SetPosition0%"},          {"0"},          ACTID_SET_CURPOS_0,              MODIFIER_CTRL,     0},
  /* NOTE: these used to be "... to 10% of ..." etc. but msgmerge seems not to like such "c-format". */
  { N_("set position to 1/10 of current stream"),
    {"SetPosition10%"},         {"1"},          ACTID_SET_CURPOS_10,             MODIFIER_CTRL,     0},
  { N_("set position to 2/10 of current stream"),
    {"SetPosition20%"},         {"2"},          ACTID_SET_CURPOS_20,             MODIFIER_CTRL,     0},
  { N_("set position to 3/10 of current stream"),
    {"SetPosition30%"},         {"3"},          ACTID_SET_CURPOS_30,             MODIFIER_CTRL,     0},
  { N_("set position to 4/10 of current stream"),
    {"SetPosition40%"},         {"4"},          ACTID_SET_CURPOS_40,             MODIFIER_CTRL,     0},
  { N_("set position to 5/10 of current stream"),
    {"SetPosition50%"},         {"5"},          ACTID_SET_CURPOS_50,             MODIFIER_CTRL,     0},
  { N_("set position to 6/10 of current stream"),
    {"SetPosition60%"},         {"6"},          ACTID_SET_CURPOS_60,             MODIFIER_CTRL,     0},
  { N_("set position to 7/10 of current stream"),
    {"SetPosition70%"},         {"7"},          ACTID_SET_CURPOS_70,             MODIFIER_CTRL,     0},
  { N_("set position to 8/10 of current stream"),
    {"SetPosition80%"},         {"8"},          ACTID_SET_CURPOS_80,             MODIFIER_CTRL,     0},
  { N_("set position to 9/10 of current stream"),
    {"SetPosition90%"},         {"9"},          ACTID_SET_CURPOS_90,             MODIFIER_CTRL,     0},
  { N_("set position to end of current stream"),
    {"SetPosition100%"},        {"End"},        ACTID_SET_CURPOS_100,            MODIFIER_CTRL,     0},
  { N_("increment playback speed"),
    {"SpeedFaster"},            {"Up"},         ACTID_SPEED_FAST,                MODIFIER_NOMOD,    0},
  { N_("decrement playback speed"),
    {"SpeedSlower"},            {"Down"},       ACTID_SPEED_SLOW,                MODIFIER_NOMOD,    0},
  { N_("reset playback speed"),
    {"SpeedReset"},             {"Down"},       ACTID_SPEED_RESET,               MODIFIER_META,     0},
  { N_("increment audio volume"),
    {"Volume+"},                {"V"},          ACTID_pVOLUME,                   MODIFIER_NOMOD,    0},
  { N_("decrement audio volume"),
    {"Volume-"},                {"v"},          ACTID_mVOLUME,                   MODIFIER_NOMOD,    0},
  { N_("increment amplification level"),
    {"Amp+"},                   {"V"},          ACTID_pAMP,                      MODIFIER_CTRL,     0},
  { N_("decrement amplification level"),
    {"Amp-"},                   {"v"},          ACTID_mAMP,                      MODIFIER_CTRL,     0},
  { N_("reset amplification to default value"),
    {"ResetAmp"},               {"A"},          ACTID_AMP_RESET,                 MODIFIER_CTRL,     0},
  { N_("audio muting toggle"),
    {"Mute"},                   {"m"},          ACTID_MUTE,                      MODIFIER_CTRL,     0},
  { N_("select next audio channel"),
    {"AudioChannelNext"},       {"plus"},       ACTID_AUDIOCHAN_NEXT,            MODIFIER_NOMOD,    0},
  { N_("select previous audio channel"),
    {"AudioChannelPrior"},      {"minus"},      ACTID_AUDIOCHAN_PRIOR,           MODIFIER_NOMOD,    0},
  { N_("visibility toggle of audio post effect window"),
    {"APProcessShow"},          {"VOID"},       ACTID_APP,                       MODIFIER_NOMOD,    KBE_FLAG_gui | KBE_FLAG_void},
  { N_("toggle post effect usage"),
    {"APProcessEnable"},        {"VOID"},       ACTID_APP_ENABLE,                MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("select next sub picture (subtitle) channel"),
    {"SpuNext"},                {"period"},     ACTID_SPU_NEXT,                  MODIFIER_NOMOD,    0},
  { N_("select previous sub picture (subtitle) channel"),
    {"SpuPrior"},               {"comma"},      ACTID_SPU_PRIOR,                 MODIFIER_NOMOD,    0},
  { N_("interlaced mode toggle"),
    {"ToggleInterleave"},       {"i"},          ACTID_TOGGLE_INTERLEAVE,         MODIFIER_NOMOD,    0},
  { N_("cycle aspect ratio values"),
    {"ToggleAspectRatio"},      {"a"},          ACTID_TOGGLE_ASPECT_RATIO,       MODIFIER_NOMOD,    0},
  { N_("reduce the output window size by factor 1.2"),
    {"WindowReduce"},           {"less"},       ACTID_WINDOWREDUCE,              MODIFIER_NOMOD,    0},
  { N_("enlarge the output window size by factor 1.2"),
    {"WindowEnlarge"},          {"greater"},    ACTID_WINDOWENLARGE,             MODIFIER_NOMOD,    0},
  { N_("set video output window to 50%"),
    {"Window50"},               {"1"},          ACTID_WINDOW50,                  MODIFIER_META,     0},
  { N_("set video output window to 100%"),
    {"Window100"},              {"2"},          ACTID_WINDOW100,                 MODIFIER_META,     0},
  { N_("set video output window to 200%"),
    {"Window200"},              {"3"},          ACTID_WINDOW200,                 MODIFIER_META,     0},
  { N_("zoom in"),
    {"ZoomIn"},                 {"z"},          ACTID_ZOOM_IN,                   MODIFIER_NOMOD,    0},
  { N_("zoom out"),
    {"ZoomOut"},                {"Z"},          ACTID_ZOOM_OUT,                  MODIFIER_NOMOD,    0},
  { N_("zoom in horizontally"),
    {"ZoomInX"},                {"z"},          ACTID_ZOOM_X_IN,                 MODIFIER_CTRL,     0},
  { N_("zoom out horizontally"),
    {"ZoomOutX"},               {"Z"},          ACTID_ZOOM_X_OUT,                MODIFIER_CTRL,     0},
  { N_("zoom in vertically"),
    {"ZoomInY"},                {"z"},          ACTID_ZOOM_Y_IN,                 MODIFIER_META,     0},
  { N_("zoom out vertically"),
    {"ZoomOutY"},               {"Z"},          ACTID_ZOOM_Y_OUT,                MODIFIER_META,     0},
  { N_("reset zooming"),
    {"ZoomReset"},              {"z"},          ACTID_ZOOM_RESET,                MODIFIER_CTRL | MODIFIER_META, 0},
  { N_("zoom move left"),
    {"ZoomLeft"},               {"Left"},       ACTID_ZOOM_LEFT,                 MODIFIER_MOD5,     0},
  { N_("zoom move right"),
    {"ZoomRight"},              {"Right"},      ACTID_ZOOM_RIGHT,                MODIFIER_MOD5,     0},
  { N_("zoom move up"),
    {"ZoomUp"},                 {"Up"},         ACTID_ZOOM_UP,                   MODIFIER_MOD5,     0},
  { N_("zoom move down"),
    {"ZoomDown"},               {"Down"},       ACTID_ZOOM_DOWN,                 MODIFIER_MOD5,     0},
  { N_("resize output window to stream size"),
    {"Zoom1:1"},                {"s"},          ACTID_ZOOM_1_1,                  MODIFIER_NOMOD,    0},
  { N_("flip video sideways"),
    {"FlipH"},                  {"f"},          ACTID_FLIP_H,                    MODIFIER_META,     0},
  { N_("flip video upside down"),
    {"FlipV"},                  {"F"},          ACTID_FLIP_V,                    MODIFIER_META,     0},
  { N_("fullscreen toggle"),
    {"ToggleFullscreen"},       {"f"},          ACTID_TOGGLE_FULLSCREEN,         MODIFIER_NOMOD,    0},
#ifdef HAVE_XINERAMA
  { N_("Xinerama fullscreen toggle"),
    {"ToggleXineramaFullscr"},  {"F"},          ACTID_TOGGLE_XINERAMA_FULLSCREEN,MODIFIER_NOMOD,    0},
#endif
  { N_("jump to media Menu"),
    {"Menu"},                   {"Escape"},     ACTID_EVENT_MENU1,               MODIFIER_NOMOD,    0},
  { N_("jump to Title Menu"),
    {"TitleMenu"},              {"F1"},         ACTID_EVENT_MENU2,               MODIFIER_NOMOD,    0},
  { N_("jump to Root Menu"),
    {"RootMenu"},               {"F2"},         ACTID_EVENT_MENU3,               MODIFIER_NOMOD,    0},
  { N_("jump to Subpicture Menu"),
    {"SubpictureMenu"},         {"F3"},         ACTID_EVENT_MENU4,               MODIFIER_NOMOD,    0},
  { N_("jump to Audio Menu"),
    {"AudioMenu"},              {"F4"},         ACTID_EVENT_MENU5,               MODIFIER_NOMOD,    0},
  { N_("jump to Angle Menu"),
    {"AngleMenu"},              {"F5"},         ACTID_EVENT_MENU6,               MODIFIER_NOMOD,    0},
  { N_("jump to Part Menu"),
    {"PartMenu"},               {"F6"},         ACTID_EVENT_MENU7,               MODIFIER_NOMOD,    0},
  { N_("menu navigate up"),
    {"EventUp"},                {"KP_Up"},      ACTID_EVENT_UP,                  MODIFIER_NOMOD,    0},
  { N_("menu navigate down"),
    {"EventDown"},              {"KP_Down"},    ACTID_EVENT_DOWN,                MODIFIER_NOMOD,    0},
  { N_("menu navigate left"),
    {"EventLeft"},              {"KP_Left"},    ACTID_EVENT_LEFT,                MODIFIER_NOMOD,    0},
  { N_("menu navigate right"),
    {"EventRight"},             {"KP_Right"},   ACTID_EVENT_RIGHT,               MODIFIER_NOMOD,    0},
  { N_("menu select"),
    {"EventSelect"},            {"KP_Enter"},   ACTID_EVENT_SELECT,              MODIFIER_NOMOD,    0},
  { N_("jump to next chapter"),
    {"EventNext"},              {"KP_Next"},    ACTID_EVENT_NEXT,                MODIFIER_NOMOD,    0},
  { N_("jump to previous chapter"),
    {"EventPrior"},             {"KP_Prior"},   ACTID_EVENT_PRIOR,               MODIFIER_NOMOD,    0},
  { N_("select next angle"),
    {"EventAngleNext"},         {"KP_Home"},    ACTID_EVENT_ANGLE_NEXT,          MODIFIER_NOMOD,    0},
  { N_("select previous angle"),
    {"EventAnglePrior"},        {"KP_End"},     ACTID_EVENT_ANGLE_PRIOR,         MODIFIER_NOMOD,    0},
  { N_("visibility toggle of help window"),
    {"HelpShow"},               {"h"},          ACTID_HELP_SHOW,                 MODIFIER_META,     KBE_FLAG_gui},
  { N_("visibility toggle of video post effect window"),
    {"VPProcessShow"},          {"P"},          ACTID_VPP,                       MODIFIER_META,     KBE_FLAG_gui},
  { N_("toggle post effect usage"),
    {"VPProcessEnable"},        {"P"},          ACTID_VPP_ENABLE,                MODIFIER_CTRL | MODIFIER_META, 0},
  { N_("visibility toggle of output window"),
    {"ToggleWindowVisibility"}, {"h"},          ACTID_TOGGLE_WINOUT_VISIBLITY,   MODIFIER_NOMOD,    KBE_FLAG_gui},
  { N_("bordered window toggle of output window"),
    {"ToggleWindowBorder"},     {"b"},          ACTID_TOGGLE_WINOUT_BORDER,      MODIFIER_NOMOD,    KBE_FLAG_gui},
  { N_("visibility toggle of UI windows"),
    {"ToggleVisibility"},       {"g"},          ACTID_TOGGLE_VISIBLITY,          MODIFIER_NOMOD,    KBE_FLAG_gui},
  { N_("visibility toggle of control window"),
    {"ControlShow"},            {"c"},          ACTID_CONTROLSHOW,               MODIFIER_META,     KBE_FLAG_gui},
  { N_("visibility toggle of audio control window"),
    {"AControlShow"},           {"a"},          ACTID_ACONTROLSHOW,              MODIFIER_META,     KBE_FLAG_gui},
  { N_("visibility toggle of mrl browser window"),
    {"MrlBrowser"},             {"m"},          ACTID_MRLBROWSER,                MODIFIER_META,     KBE_FLAG_gui},
  { N_("visibility toggle of playlist editor window"),
    {"PlaylistEditor"},         {"p"},          ACTID_PLAYLIST,                  MODIFIER_META,     KBE_FLAG_gui},
  { N_("visibility toggle of the setup window"),
    {"SetupShow"},              {"s"},          ACTID_SETUP,                     MODIFIER_META,     KBE_FLAG_gui},
  { N_("visibility toggle of the event sender window"),
    {"EventSenderShow"},        {"e"},          ACTID_EVENT_SENDER,              MODIFIER_META,     KBE_FLAG_gui},
  { N_("visibility toggle of analog TV window"),
    {"TVAnalogShow"},           {"t"},          ACTID_TVANALOG,                  MODIFIER_META,     KBE_FLAG_gui},
  { N_("visibility toggle of log viewer"),
    {"ViewlogShow"},            {"l"},          ACTID_VIEWLOG,                   MODIFIER_META,     KBE_FLAG_gui},
  { N_("visibility toggle of stream info window"),
    {"StreamInfosShow"},        {"i"},          ACTID_STREAM_INFOS,              MODIFIER_META,     KBE_FLAG_gui},
  { N_("display stream information using OSD"),
    {"OSDStreamInfos"},         {"i"},          ACTID_OSD_SINFOS,                MODIFIER_CTRL,     0},
  { N_("display information using OSD"),
    {"OSDWriteText"},           {"VOID"},       ACTID_OSD_WTEXT,                 MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("show OSD menu"),
    {"OSDMenu"},                {"O"},          ACTID_OSD_MENU,                  MODIFIER_NOMOD,    0},
  { N_("enter key binding editor"),
    {"KeyBindingEditor"},       {"k"},          ACTID_KBEDIT,                    MODIFIER_META,     KBE_FLAG_gui},
  { N_("enable key bindings (not useful to bind a key to it!)"),
    {"KeyBindingsEnable"},      {"VOID"},       ACTID_KBENABLE,                  MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("open file selector"),
    {"FileSelector"},           {"o"},          ACTID_FILESELECTOR,              MODIFIER_CTRL,     KBE_FLAG_gui},
  { N_("select a subtitle file"),
    {"SubSelector"},            {"S"},          ACTID_SUBSELECT,                 MODIFIER_CTRL,     KBE_FLAG_gui},
#ifdef HAVE_CURL
  { N_("download a skin from the skin server"),
    {"SkinDownload"},           {"d"},          ACTID_SKINDOWNLOAD,              MODIFIER_CTRL,     KBE_FLAG_gui},
#endif
  { N_("grab pointer toggle"),
    {"GrabPointer"},            {"Insert"},     ACTID_GRAB_POINTER,              MODIFIER_NOMOD,    KBE_FLAG_gui},
  { N_("enter the number 0"),
    {"Number0"},                {"0"},          ACTID_EVENT_NUMBER_0,            MODIFIER_NOMOD,    0},
  { N_("enter the number 1"),
    {"Number1"},                {"1"},          ACTID_EVENT_NUMBER_1,            MODIFIER_NOMOD,    0},
  { N_("enter the number 2"),
    {"Number2"},                {"2"},          ACTID_EVENT_NUMBER_2,            MODIFIER_NOMOD,    0},
  { N_("enter the number 3"),
    {"Number3"},                {"3"},          ACTID_EVENT_NUMBER_3,            MODIFIER_NOMOD,    0},
  { N_("enter the number 4"),
    {"Number4"},                {"4"},          ACTID_EVENT_NUMBER_4,            MODIFIER_NOMOD,    0},
  { N_("enter the number 5"),
    {"Number5"},                {"5"},          ACTID_EVENT_NUMBER_5,            MODIFIER_NOMOD,    0},
  { N_("enter the number 6"),
    {"Number6"},                {"6"},          ACTID_EVENT_NUMBER_6,            MODIFIER_NOMOD,    0},
  { N_("enter the number 7"),
    {"Number7"},                {"7"},          ACTID_EVENT_NUMBER_7,            MODIFIER_NOMOD,    0},
  { N_("enter the number 8"),
    {"Number8"},                {"8"},          ACTID_EVENT_NUMBER_8,            MODIFIER_NOMOD,    0},
  { N_("enter the number 9"),
    {"Number9"},                {"9"},          ACTID_EVENT_NUMBER_9,            MODIFIER_NOMOD,    0},
  { N_("add 10 to the next entered number"),
    {"Number10add"},            {"plus"},       ACTID_EVENT_NUMBER_10_ADD,       MODIFIER_MOD3,     0},
  { N_("set position in current stream to numeric percentage"),
    {"SetPosition%"},           {"slash"},      ACTID_SET_CURPOS,                MODIFIER_NOMOD,    0},
  { N_("set position forward by numeric argument in current stream"),
    {"SeekRelative+"},          {"Up"},         ACTID_SEEK_REL_p,                MODIFIER_META,     0},
  { N_("set position back by numeric argument in current stream"),
    {"SeekRelative-"},          {"Up"},         ACTID_SEEK_REL_m,                MODIFIER_MOD3,     0},
  { N_("change audio video syncing (delay video)"),
    {"AudioVideoDecay+"},       {"m"},          ACTID_AV_SYNC_p3600,             MODIFIER_NOMOD,    0},
  { N_("change audio video syncing (delay audio)"),
    {"AudioVideoDecay-"},       {"n"},          ACTID_AV_SYNC_m3600,             MODIFIER_NOMOD,    0},
  { N_("reset audio video syncing offset"),
    {"AudioVideoDecayReset"},   {"Home"},       ACTID_AV_SYNC_RESET,             MODIFIER_NOMOD,    0},
  { N_("change subtitle syncing (delay video)"),
    {"SpuVideoDecay+"},         {"M"},          ACTID_SV_SYNC_p,                 MODIFIER_NOMOD,    0},
  { N_("change subtitle syncing (delay subtitles)"),
    {"SpuVideoDecay-"},         {"N"},          ACTID_SV_SYNC_m,                 MODIFIER_NOMOD,    0},
  { N_("reset subtitle syncing offset"),
    {"SpuVideoDecayReset"},     {"End"},        ACTID_SV_SYNC_RESET,             MODIFIER_NOMOD,    0},
  { N_("toggle TV modes (on the DXR3)"),
    {"ToggleTVmode"},           {"o"},          ACTID_TOGGLE_TVMODE,             MODIFIER_CTRL | MODIFIER_META, 0},
  { N_("switch Monitor to DPMS standby mode"),
    {"DPMSStandby"},            {"d"},          ACTID_DPMSSTANDBY,               MODIFIER_NOMOD,    0},
  { N_("increase hue by 10"),
    {"HueControl+"},            {"VOID"},       ACTID_HUECONTROLp,               MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("decrease hue by 10"),
    {"HueControl-"},            {"VOID"},       ACTID_HUECONTROLm,               MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("increase saturation by 10"),
    {"SaturationControl+"},     {"VOID"},       ACTID_SATURATIONCONTROLp,        MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("decrease saturation by 10"),
    {"SaturationControl-"},     {"VOID"},       ACTID_SATURATIONCONTROLm,        MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("increase brightness by 10"),
    {"BrightnessControl+"},     {"VOID"},       ACTID_BRIGHTNESSCONTROLp,        MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("decrease brightness by 10"),
    {"BrightnessControl-"},     {"VOID"},       ACTID_BRIGHTNESSCONTROLm,        MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("increase contrast by 10"),
    {"ContrastControl+"},       {"VOID"},       ACTID_CONTRASTCONTROLp,          MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("decrease contrast by 10"),
    {"ContrastControl-"},       {"VOID"},       ACTID_CONTRASTCONTROLm,          MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("increase gamma by 10"),
    {"GammaControl+"},          {"VOID"},       ACTID_GAMMACONTROLp,             MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("decrease gamma by 10"),
    {"GammaControl-"},          {"VOID"},       ACTID_GAMMACONTROLm,             MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("increase sharpness by 10"),
    {"SharpnessControl+"},      {"VOID"},       ACTID_SHARPNESSCONTROLp,         MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("decrease sharpness by 10"),
    {"SharpnessControl-"},      {"VOID"},       ACTID_SHARPNESSCONTROLm,         MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("increase noise reduction by 10"),
    {"NoiseReductionControl+"}, {"VOID"},       ACTID_NOISEREDUCTIONCONTROLp,    MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("decrease noise reduction by 10"),
    {"NoiseReductionControl-"}, {"VOID"},       ACTID_NOISEREDUCTIONCONTROLm,    MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("quit the program"),
    {"Quit"},                   {"q"},          ACTID_QUIT,                      MODIFIER_NOMOD,    0},
  { N_("input_pvr: set input"),
    {"PVRSetInput"},            {"VOID"},       ACTID_PVR_SETINPUT,              MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("input_pvr: set frequency"),
    {"PVRSetFrequency"},        {"VOID"},       ACTID_PVR_SETFREQUENCY,          MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("input_pvr: mark the start of a new stream section"),
    {"PVRSetMark"},             {"VOID"},       ACTID_PVR_SETMARK,               MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("input_pvr: set the name for the current stream section"),
    {"PVRSetName"},             {"VOID"},       ACTID_PVR_SETNAME,               MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("input_pvr: save the stream section"),
    {"PVRSave"},                {"VOID"},       ACTID_PVR_SAVE,                  MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("Open playlist"),
    {"PlaylistOpen"},           {"VOID"},       ACTID_PLAYLIST_OPEN,             MODIFIER_NOMOD,    KBE_FLAG_void},
#ifdef ENABLE_VDR_KEYS
  { N_("VDR Red button"),
    {"VDRButtonRed"},           {"VOID"},       ACTID_EVENT_VDR_RED,             MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("VDR Green button"),
    {"VDRButtonGreen"},         {"VOID"},       ACTID_EVENT_VDR_GREEN,           MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("VDR Yellow button"),
    {"VDRButtonYellow"},        {"VOID"},       ACTID_EVENT_VDR_YELLOW,          MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("VDR Blue button"),
    {"VDRButtonBlue"},          {"VOID"},       ACTID_EVENT_VDR_BLUE,            MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("VDR Play"),
    {"VDRPlay"},                {"VOID"},       ACTID_EVENT_VDR_PLAY,            MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("VDR Pause"),
    {"VDRPause"},               {"VOID"},       ACTID_EVENT_VDR_PAUSE,           MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("VDR Stop"),
    {"VDRStop"},                {"VOID"},       ACTID_EVENT_VDR_STOP,            MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("VDR Record"),
    {"VDRRecord"},              {"VOID"},       ACTID_EVENT_VDR_RECORD,          MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("VDR Fast forward"),
    {"VDRFastFwd"},             {"VOID"},       ACTID_EVENT_VDR_FASTFWD,         MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("VDR Fast rewind"),
    {"VDRFastRew"},             {"VOID"},       ACTID_EVENT_VDR_FASTREW,         MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("VDR Power"),
    {"VDRPower"},               {"VOID"},       ACTID_EVENT_VDR_POWER,           MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("VDR Channel +"),
    {"VDRChannelPlus"},         {"VOID"},       ACTID_EVENT_VDR_CHANNELPLUS,     MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("VDR Channel -"),
    {"VDRChannelMinus"},        {"VOID"},       ACTID_EVENT_VDR_CHANNELMINUS,    MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("VDR Schedule menu"),
    {"VDRSchedule"},            {"VOID"},       ACTID_EVENT_VDR_SCHEDULE,        MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("VDR Channel menu"),
    {"VDRChannels"},            {"VOID"},       ACTID_EVENT_VDR_CHANNELS,        MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("VDR Timers menu"),
    {"VDRTimers"},              {"VOID"},       ACTID_EVENT_VDR_TIMERS,          MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("VDR Recordings menu"),
    {"VDRRecordings"},          {"VOID"},       ACTID_EVENT_VDR_RECORDINGS,      MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("VDR Setup menu"),
    {"VDRSetup"},               {"VOID"},       ACTID_EVENT_VDR_SETUP,           MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("VDR Command menu"),
    {"VDRCommands"},            {"VOID"},       ACTID_EVENT_VDR_COMMANDS,        MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("VDR Command back"),
    {"VDRBack"},                {"VOID"},       ACTID_EVENT_VDR_BACK,            MODIFIER_NOMOD,    KBE_FLAG_void},
#ifdef XINE_EVENT_VDR_USER0 /* #ifdef is precaution for backward compatibility at the moment */
  { N_("VDR User command 0"),
    {"VDRUser0"},               {"VOID"},       ACTID_EVENT_VDR_USER0,           MODIFIER_NOMOD,    KBE_FLAG_void},
#endif
  { N_("VDR User command 1"),
    {"VDRUser1"},               {"VOID"},       ACTID_EVENT_VDR_USER1,           MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("VDR User command 2"),
    {"VDRUser2"},               {"VOID"},       ACTID_EVENT_VDR_USER2,           MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("VDR User command 3"),
    {"VDRUser3"},               {"VOID"},       ACTID_EVENT_VDR_USER3,           MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("VDR User command 4"),
    {"VDRUser4"},               {"VOID"},       ACTID_EVENT_VDR_USER4,           MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("VDR User command 5"),
    {"VDRUser5"},               {"VOID"},       ACTID_EVENT_VDR_USER5,           MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("VDR User command 6"),
    {"VDRUser6"},               {"VOID"},       ACTID_EVENT_VDR_USER6,           MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("VDR User command 7"),
    {"VDRUser7"},               {"VOID"},       ACTID_EVENT_VDR_USER7,           MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("VDR User command 8"),
    {"VDRUser8"},               {"VOID"},       ACTID_EVENT_VDR_USER8,           MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("VDR User command 9"),
    {"VDRUser9"},               {"VOID"},       ACTID_EVENT_VDR_USER9,           MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("VDR Volume +"),
    {"VDRVolumePlus"},          {"VOID"},       ACTID_EVENT_VDR_VOLPLUS,         MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("VDR Volume -"),
    {"VDRVolumeMinus"},         {"VOID"},       ACTID_EVENT_VDR_VOLMINUS,        MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("VDR Mute audio"),
    {"VDRMute"},                {"VOID"},       ACTID_EVENT_VDR_MUTE,            MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("VDR Audio menu"),
    {"VDRAudio"},               {"VOID"},       ACTID_EVENT_VDR_AUDIO,           MODIFIER_NOMOD,    KBE_FLAG_void},
  { N_("VDR Command info"),
    {"VDRInfo"},                {"VOID"},       ACTID_EVENT_VDR_INFO,            MODIFIER_NOMOD,    KBE_FLAG_void},
#ifdef XINE_EVENT_VDR_CHANNELPREVIOUS /* #ifdef is precaution for backward compatibility at the moment */
  { N_("VDR Previous channel"),
    {"VDRChannelPrevious"},     {"VOID"},       ACTID_EVENT_VDR_CHANNELPREVIOUS, MODIFIER_NOMOD,    KBE_FLAG_void},
#endif
#ifdef XINE_EVENT_VDR_SUBTITLES /* #ifdef is precaution for backward compatibility at the moment */
  { N_("VDR Subtiles menu"),
    {"VDRSubtitles"},           {"VOID"},       ACTID_EVENT_VDR_SUBTITLES,       MODIFIER_NOMOD,    KBE_FLAG_void},
#endif
#endif
};

/* for now,
 * $ tfi src/xitk/kbindings_common.c -x 1 "default_binding_table[] = {%h};" "%h" -x 0 "{ N_(" "-" -L
 * says 193. if we ever get above 255, kbinding_t.rev_id needs blown up to uint16_t. ;-) */
#define KBT_NUM_BASE (sizeof (default_binding_table) / sizeof (default_binding_table[0]))
#define _kbt_entry(kbt,index) (XITK_0_TO_MAX_MINUS_1 (index, KBT_NUM_BASE) ? kbt->base + index : kbt->alias[index - KBT_NUM_BASE])
#define KBT_REV_ID_NUM (2 * ACTID_IS_INPUT_EVENT)
#define KBT_REV_ID_MASK (KBT_REV_ID_NUM - 1)
#define KBT_KEY_IS_STATIC(_s) ((_s >= default_binding_table[0].key.s) && (_s <= default_binding_table[KBT_NUM_BASE - 1].key.s))
#define KBT_KEY_VOID (default_binding_table[KBT_NUM_BASE - 1].key.s)

static void _kbt_entry_set_void (kbinding_entry_t *e) {
  e->key = KBT_KEY_VOID;
  e->modifier = 0;
  e->flags |= KBE_FLAG_void;
}

static const char *refs_ref (const char *refs) {
  if (!KBT_KEY_IS_STATIC (refs)) {
    union {const char *s; uint8_t *u;} s;
    s.s = refs;
    if (!s.u[-1])
      return KBT_KEY_VOID;
    s.u[-1]++;
#ifdef _KBT_DEBUG_REFS
    printf ("gui.kbindings.key.ref (%s, %d).\n", s.s, (int)s.u[-1]);
#endif
  }
  return refs;
}

/* return freed. */
static int refs_unref (const char **refs) {
  if (!KBT_KEY_IS_STATIC (*refs)) {
    union {const char *s; uint8_t *u;} s;
    s.s = *refs;
    if (s.u[-1]) {
      s.u[-1]--;
#ifdef _KBT_DEBUG_REFS
      printf ("gui.kbindings.key.unref (%s, %d).\n", s.s, (int)s.u[-1]);
#endif
      if (!s.u[-1]) {
        free (s.u - 4);
        *refs = NULL;
        return 1;
      }
    }
  } else {
#ifdef _KBT_DEBUG_REFS
    printf ("gui.kbindings.key.unref (%s, static).\n", *refs);
#endif
  }
  *refs = NULL;
  return 0;
}

struct kbinding_s {
  gGui_t           *gui;
  const char       *mod_names[5];
  int               num_entries;
  xine_sarray_t    *action_index, *key_index;
  kbinding_entry_t *last, base[KBT_NUM_BASE], *alias[KBT_MAX_ENTRIES - KBT_NUM_BASE];
  uint8_t           rev_id[KBT_REV_ID_NUM];
};

static int _kbt_action_cmp (void *a, void *b) {
  kbinding_entry_t *d = (kbinding_entry_t *)a;
  kbinding_entry_t *e = (kbinding_entry_t *)b;
  union {const char *s; const int32_t *v;} v1, v2;
  v1.s = d->name;
  v2.s = e->name;
  int f;
  /* 'A' == 'a, diff is more likely than equal here. */
  if ((f = (v1.v[0] | 0x20202020) - (v2.v[0] | 0x20202020)))
    return f;
  if (v1.s != v2.s) {
    /* strings are 6 longwords at most, so safety stop there. */
    if ((f = (v1.v[1] | 0x20202020) - (v2.v[1] | 0x20202020)))
      return f;
    if ((f = (v1.v[2] | 0x20202020) - (v2.v[2] | 0x20202020)))
      return f;
    if ((f = (v1.v[3] | 0x20202020) - (v2.v[3] | 0x20202020)))
      return f;
    if ((f = (v1.v[4] | 0x20202020) - (v2.v[4] | 0x20202020)))
      return f;
    if ((f = (v1.v[5] | 0x20202020) - (v2.v[5] | 0x20202020)))
      return f;
    /* little optimization: redirect alias dummy names to the found original. */
    d->name = v2.s;
  }
  return (int)(d->flags & KBE_FLAG_alias) - (int)(e->flags & KBE_FLAG_alias);
}

#if defined (XINE_SARRAY) && (XINE_SARRAY >= 3)
static unsigned int _kbt_action_hash (void *a) {
  kbinding_entry_t *d = (kbinding_entry_t *)a;
  union {const char *s; const int32_t *v;} v1;
  /* hash_quality = 87.8%. */
  v1.s = d->name;
  uint32_t v = ((v1.v[0] & 0x1f1f1f1f) ^ 0x030c1106)
             + ((v1.v[1] & 0x1f1f1f1f) ^ 0x050a0906)
             + ((v1.v[2] & 0x1f1f1f1f) ^ 0x04010802)
             +  (v1.v[3] & 0x1f1f1f1f)
             +  (v1.v[4] & 0x1f1f1f1f)
             +  (v1.v[5] & 0x1f1f1f1f);
  v += (v >> 5) & 0x07070707;
  v &= 0x1f1f1f1f;
  v += v >> 15;
  v += (v >> 6) & 0x0303;
  v &= 0x3f3f;
  v += v >> 7;
  v += (v >> 6) & 0x03;
  return v & 0x3f;
}

static unsigned int _kbt_key_hash (void *a) {
  static const uint8_t tab[256] = {
     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
     0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 0, 0, 0, 0, 0, 0,
     0,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,
    25,26,27,28,29,30,31,32,33,34,35, 0, 0, 0, 0, 0,
     0,36,37,38,39,40,41,42,43,44,45,46,47,48,49,50,
    51,52,53,54,55,56,57,58,59,60,61, 0, 0, 0, 0, 0,
     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
     0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
  };
  kbinding_entry_t *d = (kbinding_entry_t *)a;
  const uint8_t *s = (const uint8_t *)d->key;
  return tab[s[0]];
}
#endif

static int _kbt_key_cmp (void *a, void *b) {
  /* yes this depends on string length and mmachine endian,
   * but it is still a atable sort. */
  kbinding_entry_t *d = (kbinding_entry_t *)a;
  kbinding_entry_t *e = (kbinding_entry_t *)b;
  union {const char *s; uint8_t *u; const int32_t *v;} v1, v2;
  const union {char s[4]; int32_t v;} last_mask = {{0, 0, 0, 255}};
  /* "A" != "a" */
  v1.s = d->key;
  v2.s = e->key;
  /* equal is very likely here due to the hash. */
  if (v1.v != v2.v) {
    int f;
    /* first longword cannot be the safety tail. */
    if ((f = *v1.v - *v2.v))
      return f;
    /* ok, test the rest. */
    do {
      v1.v++, v2.v++;
    } while (!(f = *v1.v - *v2.v));
    /* at this point, we hit either a real diff, or the safety tail. */
    if ((v1.v[-1] | v2.v[-1]) & last_mask.v)
      return f;
    /* we hit the safety tail, and we got duplicate strings here, unify them. */
    v1.s = d->key;
    v2.s = e->key;
    /* NOTE: xine_sarray_binary_search () calls foo_cmp (new_entry, array_member). */
    if (KBT_KEY_IS_STATIC (v2.s)) {
#ifdef _KBT_DEBUG_REFS
      printf ("gui.kbindings.key.unify (%s, %p -> %p, static).\n", v1.s, (const void *)v1.s, (const void *)v2.s);
#endif
    } else {
      if (v2.u[-1])
        v2.u[-1]++;
#ifdef _KBT_DEBUG_REFS
      printf ("gui.kbindings.key.unify (%s, %p -> %p, %d).\n", v1.s, (const void *)v1.s, (const void *)v2.s, (int)v2.u[-1]);
#endif
    }
    refs_unref (&d->key);
    d->key = v2.s;
  }
  return (int)d->modifier - (int)e->modifier;
}

static void _kbt_entry_test_default (kbinding_entry_t *e, unsigned int idx) {
  const union {char s[4]; int32_t v;} last_mask = {{0, 0, 0, 255}};
  union {const char *s; const int32_t *v;} v1;
  const int32_t *v2 = default_binding_table[idx].key.v;
  /* "A" != "a" */
  v1.s = e->key;
  e->flags &= ~KBE_FLAG_default;
  if (v1.v != v2) {
    if (*v1.v != *v2)
      return;
    do {
      v1.v++, v2++;
    } while (*v1.v == *v2);
    if ((v1.v[-1] | v2[-1]) & last_mask.v)
      return;
  }
  if (e->modifier != default_binding_table[idx].modifier)
    return;
  e->flags |= KBE_FLAG_default;
}

static void _kbt_index_add (kbinding_t *kbt, kbinding_entry_t *entry, int with_action) {
  if (with_action)
    xine_sarray_add (kbt->action_index, entry);
  if (!(entry->flags & KBE_FLAG_void))
    xine_sarray_add (kbt->key_index, entry);
}

static void _kbt_index_remove (kbinding_t *kbt, kbinding_entry_t *entry, int with_action) {
#ifdef XINE_SARRAY_MODE_DEFAULT
  if (with_action)
    xine_sarray_remove_ptr (kbt->action_index, entry);
  if (!(entry->flags & KBE_FLAG_void))
    xine_sarray_remove_ptr (kbt->key_index, entry);
#else
  int i;
  if (with_action) {
    i = xine_sarray_binary_search (kbt->action_index, entry);
    if (i >= 0)
      xine_sarray_remove (kbt->action_index, i);
  }
  if (!(entry->flags & KBE_FLAG_void)) {
    i = xine_sarray_binary_search (kbt->key_index, entry);
    if (i >= 0)
      xine_sarray_remove (kbt->key_index, i);
  }
#endif
}

/*
 * Return a key binding entry (if available) matching with action string.
 */
static ATTR_INLINE_ALL_STRINGOPS kbinding_entry_t *_kbt_lookup_action (kbinding_t *kbt, const char *action) {
  union {
    int32_t i[6];
    char s[6 * 4];
  } buf;
  kbinding_entry_t dummy;
  size_t s = strlen (action) + 1;
  int i;

  if (s > sizeof (buf))
    s = sizeof (buf);
  for (i = s >> 2; i < (int)sizeof (buf) / 4; i++)
    buf.i[i] = 0;
  memcpy (buf.s, action, s);
  dummy.name = buf.s;
  dummy.flags = 0;
  i = xine_sarray_binary_search (kbt->action_index, &dummy);
  return (kbinding_entry_t *)xine_sarray_get (kbt->action_index, i);
}

const kbinding_entry_t *kbindings_entry_from_name (kbinding_t *kbt, const char *action) {
  uintptr_t n;

  if (!action || !kbt)
    return NULL;
  n = (uintptr_t)action;
  if (n < KBT_REV_ID_NUM) {
    n = kbt->rev_id[n];
    return (n < KBT_NUM_BASE) ? kbt->base + n : NULL;
  }
  return _kbt_lookup_action (kbt, action);
}

static char *_kbt_set_dummy_default (refs_dummy_t *dummy, const default_binding_t *b) {
  size_t w;
  const union {uint8_t u[4]; uint32_t v;} mul = {{0, 0, 1, 0}};
  union {const char *s; const int32_t *v;} v1;
  union {char *s; int32_t *v;} v2;

  v1.v =  b->key.v;
  v2.v = dummy->i + 1;
  do {
    *v2.v = *v1.v;
    v1.v++; v2.v++;
  } while (v1.s[-1]);
  w = v2.v - (dummy->i + 1);
  *v2.v = ~0u;
  dummy->i[0] = w * mul.v;
  return dummy->s + 4;
}

static int _kbt_set_dummy_key (kbinding_t *kbt, kbinding_entry_t *e) {
  union {const char *s; uint8_t *u; int32_t *v;} v1, v2;
  const union { char s[4]; int32_t u; } _void = {"void"};

#ifdef _KBT_DEBUG_REFS
  printf ("gui.kbindings.key.dummy (%s).\n", e->key);
#endif
  v1.s = e->key - 4;
  if ((v1.v[1] | 0x20202020) != _void.u) {
    int i = xine_sarray_add (kbt->key_index, e);
    e->flags &= ~KBE_FLAG_void;
    if (e->key != v1.s + 4)
      return i;
    v2.u = malloc (v1.u[2] * 4 + 8);
    if (v2.u) {
      memcpy (v2.u, v1.s, v1.u[2] * 4 + 8);
      v2.v[v2.u[2] + 1] = 0;
      v2.u[3] = 1; /* 1 ref */
      e->key = v2.s + 4;
      if (kbt->gui->verbosity >= 2)
        printf ("gui.kbindings.key.new (%s).\n", e->key);
      return i;
    }
    xine_sarray_remove (kbt->key_index, i);
  }
  _kbt_entry_set_void (e);
  return -1;
}

/* dummy != "VOID", return -1 (ok), >= 0 (index that already uses this key). */
static int _kbt_change_dummy_key (kbinding_t *kbt, kbinding_entry_t *e, refs_dummy_t *dummy, int modifier) {
  kbinding_entry_t e2;
  int i;

  e2.key = dummy->s + 4;
  e2.modifier = modifier;
  /* NOTE: this may unify away a found non void key, then bail the not found modifier. */
  i = xine_sarray_binary_search (kbt->key_index, &e2);
  if (i >= 0) {
    refs_unref (&e2.key);
    e = xine_sarray_get (kbt->key_index, i);
    return e->index;
  }
  _kbt_index_remove (kbt, e, 0);
  refs_unref (&e->key);

  e->key = e2.key;
  e->modifier = modifier;
  if (e->key == dummy->s + 4) {
    _kbt_set_dummy_key (kbt, e);
  } else {
    e->flags &= ~KBE_FLAG_void;
    xine_sarray_add (kbt->key_index, e);
  }
  return -1;
}

/*
 * Read and parse remap key binding file, if available.
 */
static void _kbinding_load_config(kbinding_t *kbt, const char *name) {
  size_t fsize = 1 << 20;
  char *fbuf;
  xitk_cfg_parse_t *tree;
  int i;

  fbuf = xitk_cfg_load (name, &fsize);
  if (!fbuf)
    return;

  tree = xitk_cfg_parse (fbuf, XITK_CFG_PARSE_CASE);
  if (!tree) {
    xitk_cfg_unload (fbuf);
    return;
  }

  for (i = tree[0].first_child; i; i = tree[i].next) {
    const char *sname = fbuf + tree[i].key, *entry = "", *key = "", *mod = "";
    refs_dummy_t dummy;
    kbinding_entry_t *e;
    int j, modifier;

    if (!tree[i].klen)
      continue;
    for (j = tree[i].first_child; j; j = tree[j].next) {
      const char *ename = fbuf + tree[j].key, *eval = fbuf + tree[j].value;

      switch (tree[j].klen) {
        case 3:
          if (!memcmp (ename, "key", 3))
            key = eval;
          break;
        case 5:
          if (!memcmp (ename, "entry", 5)) 
            entry = eval;
          break;
        case 8:
          if (!memcmp (ename, "modifier", 8))
            mod = eval;
          break;
        default: ;
      }
    }
    if (!key[0])
      continue;

    if ((tree[i].klen == 5) && !memcmp (sname, "Alias", 5)) {
      /* This should only happen if the keymap file was edited manually. */
      if (kbt->num_entries >= KBT_MAX_ENTRIES) {
        fprintf (stderr, _("xine-ui: Too many Alias entries in keymap file, entry ignored.\n"));
        continue;
      }
      sname = entry;
      if (!sname[0])
        continue;
    }
    e = _kbt_lookup_action (kbt, sname);
    if (!e)
      continue;

    modifier = MODIFIER_NOMOD;
    {
      /* NOTE: for [static] const, gcc references actual read only mem.
       * however, this here optimizes to a plain immediate value.
       * tested with gcc -S ;-) */
      const uint8_t *p = (const uint8_t *)mod;

      while (1) {
        static const uint8_t tab_char[256] = { [0] = 1, ['\t'] = 2, [' ']  = 2, [',']  = 4 };
        const union {uint8_t b[4]; uint32_t w;}
          _none = {"none"}, _ctrl = {"ctrl"}, _meta = {"meta"},
          _modm = {"\xff\xff\xff\xf8"}, _mod0 = {"mod0"},
          _cont = {"cont"},
          _mas3 = {"\xff\xff\xff"}, _rol_ = {"rol"}, _alt_ = {"alt"};
        union {uint8_t b[4]; uint32_t w;} v[2];
        const uint8_t *b;
        size_t l;

        while (tab_char[*p] & (2 | 4)) /* \t ' ' , */
          p++;
        if (!*p)
          break;
        b = p;
        do {
          while (!(tab_char[*p] & (1 | 2 | 4))) /* \0 \t ' ' , */
            p++;
          l = p - b;
          while (tab_char[*p] & 2) /* \t ' ' */
            p++;
        } while (!(tab_char[*p] & (1 | 4))); /* \0 , */
        memcpy (v[0].b, b, 8);
        v[0].w |= 0x20202020;
        v[1].w |= 0x20202020;
        if (l == 4) {
          static const uint8_t mods[8] = {
            [1] = MODIFIER_CTRL,
            [2] = MODIFIER_META,
            [3] = MODIFIER_MOD3,
            [4] = MODIFIER_MOD4,
            [5] = MODIFIER_MOD5
          };
          if (v[0].w == _none.w)
            modifier = MODIFIER_NOMOD;
          else if (v[0].w == _ctrl.w)
            modifier |= MODIFIER_CTRL;
          else if (v[0].w == _meta.w)
            modifier |= MODIFIER_META;
          else if ((v[0].w & _modm.w) == _mod0.w)
            modifier |= mods[v[0].b[3] & 7];
        } else if (l == 7) {
          if ((v[0].w == _cont.w) && ((v[1].w & _mas3.w) == _rol_.w))
            modifier |= MODIFIER_CTRL;
        } else if (l == 3) {
          if ((v[0].w & _mas3.w) == _alt_.w)
            modifier |= MODIFIER_META;
        }
      }
    }

    refs_set_dummy (&dummy, key);
    if (sname == entry) {
      kbinding_entry_t *n = (kbinding_entry_t *)malloc (sizeof (*n));
      if (n) {
        /* Add new entry */
        n->name       = e->name;
        n->id         = e->id;
        n->index      = kbt->num_entries;
        n->flags      = (e->flags | KBE_FLAG_alias) & ~KBE_FLAG_default;
        n->comment    = e->comment;
        xine_sarray_add (kbt->action_index, n);
        kbt->alias[kbt->num_entries - KBT_NUM_BASE] = n;
        kbt->num_entries++;
        n->modifier   = mod[0] ? modifier : e->modifier;
        n->key        = dummy.s + 4;
        _kbt_set_dummy_key (kbt, n);
      }
    } else {
      /* update base entry */
      union {const char *s; const int32_t *v;} v1, v2;
      const union {char s[4]; int32_t v;} last_mask = {{0, 0, 0, 255}};
      int f;
      /* "A" != "a" */
      v1.s = e->key;
      v2.s = dummy.s + 4;
      do {
        if ((f = *v1.v - *v2.v))
          break;
        do {
          v1.v++, v2.v++;
        } while (!(f = *v1.v - *v2.v));
        if (!((v1.v[-1] | v2.v[-1]) & last_mask.v))
          f = 0;
      } while (0);
      if (f) {
        _kbt_index_remove (kbt, e, 0);
        refs_unref (&e->key);
        e->flags &= ~KBE_FLAG_default;
        e->modifier   = modifier;
        e->key        = dummy.s + 4;
        _kbt_set_dummy_key (kbt, e);
      } else if (e->modifier != modifier) {
        if (!(e->flags & KBE_FLAG_void)) {
          _kbt_index_remove (kbt, e, 0);
          e->flags &= ~KBE_FLAG_default;
          e->modifier = modifier;
          xine_sarray_add (kbt->key_index, e);
        } else {
          e->flags &= ~KBE_FLAG_default;
          e->modifier = modifier;
        }
      }
    }
  }

  xitk_cfg_unparse (tree);
  xitk_cfg_unload (fbuf);
}
  
static int _kbt_reset (kbinding_t *kbt) {
  kbinding_entry_t *e;
  int idx;

  kbt->last = NULL;
  for (idx = 0; idx < (int)KBT_NUM_BASE; idx++) {
    e = kbt->base + idx;
    if (!(e->flags & KBE_FLAG_default)) {
      const char *lk = e->key;
      _kbt_index_remove (kbt, e, 0);
      e->flags = default_binding_table[idx].flags | KBE_FLAG_default;
      e->key = default_binding_table[idx].key.s;
      e->modifier = default_binding_table[idx].modifier;
      _kbt_index_add (kbt, e, 0);
      refs_unref (&lk);
    }
  }
  for (; idx < kbt->num_entries; idx++) {
    e = kbt->alias[idx - KBT_NUM_BASE];
    kbt->alias[idx - KBT_NUM_BASE] = NULL;
    _kbt_index_remove (kbt, e, 1);
    e->name = NULL;
    e->comment = NULL;
    refs_unref (&e->key);
    free (e);
  }
  kbt->num_entries = KBT_NUM_BASE;
  return -1;
}

/*
 * Check if there some redundant entries in key binding table kbt.
 */
static void _kbinding_done (void *data, int state) {
  kbinding_t *kbt = data;
  if (state == 1)
    _kbt_reset (kbt);
  else if (state == 2)
    kbedit_main (XUI_W_ON, kbt->gui);
}

static void _kbt_check_redundancy (kbinding_t *kbt) {
  uint8_t ftab[4 * KBT_MAX_ENTRIES], *t = ftab;
  int tlen = 0;

  {
    kbinding_entry_t *e1, *e2;
    int n, i;

    n = xine_sarray_size (kbt->key_index);
    e1 = xine_sarray_get (kbt->key_index, 0);
    for (i = 1; i < n; e1 = e2, i++) {
      int j1, j2;

      e2 = xine_sarray_get (kbt->key_index, i);
      if ((e1->key != e2->key) || (e1->modifier != e2->modifier))
        continue;
      j1 = (e1->name - (const char *)default_binding_table) / (int)(sizeof (default_binding_t) / sizeof (const char));
      j2 = (e2->name - (const char *)default_binding_table) / (int)(sizeof (default_binding_t) / sizeof (const char));
      if (XITK_0_TO_MAX_MINUS_1 (j1, KBT_NUM_BASE) && XITK_0_TO_MAX_MINUS_1 (j2, KBT_NUM_BASE)) {
        t[0] = j1;
        t[1] = j2;
        tlen += (t[2] = strlen (e1->comment));
        tlen += (t[3] = strlen (e2->comment));
        t += 4;
      }
    }
  }
  if (t == ftab)
    return;

  {
    const char *header = _("The following key bindings pairs are identical:\n\n"), *dna = _("and");
    const char *footer = _(".\n\nWhat do you want to do ?\n");
    size_t hlen = strlen (header), dnalen = strlen (dna), flen = strlen (footer);
    uint8_t *e = t;
    char *p, *kmsg = malloc (hlen + tlen + (e - ftab) / 4 * (dnalen + 8) + flen + 4);

    if (!kmsg)
      return;
    p = kmsg;
    memcpy (p, header, hlen); p += hlen;
    for (t = ftab; t < e; t += 4) {
      memcpy (p, "\"", 1); p += 1;
      memcpy (p, kbt->base[t[0]].comment, t[2]); p += t[2];
      memcpy (p, "\" ", 2); p += 2;
      memcpy (p, dna, dnalen); p += dnalen;
      memcpy (p, " \"", 2); p += 2;
      memcpy (p, kbt->base[t[1]].comment, t[3] + 1); p += t[3];
      memcpy (p, "\", ", 3); p += 3;
    }
    p -= 2;
    memcpy (p, "\n", 2);
    if (kbt->gui && kbt->gui->xitk) {
      if (kbt->gui->verbosity > 0)
        dump_msg (kmsg, 1);
      memcpy (p, footer, flen + 1); /* p += flen; */
      xitk_window_dialog_3 (kbt->gui->xitk, NULL,
        gui_layer_above (kbt->gui, NULL), 450, _("Keybindings error!"), _kbinding_done, kbt,
        _("Reset"), _("Editor"), _("Cancel"), NULL, 0, ALIGN_CENTER, "%s", kmsg);
    } else {
      printf ("%s", kmsg);
    }
    free (kmsg);
  }
}

/*
 * Initialize a key binding table from default, then try
 * to remap this one with (system/user) remap files.
 */

kbinding_t *kbindings_new (gGui_t *gui, const char *keymap_file) {
  kbinding_t *kbt;

  kbt = (kbinding_t *)malloc (sizeof (*kbt));
  if (!kbt)
    return NULL;

  kbt->gui = gui;
  kbt->last = NULL;
  kbt->action_index = xine_sarray_new (KBT_MAX_ENTRIES, _kbt_action_cmp);
  kbt->key_index = xine_sarray_new (KBT_MAX_ENTRIES, _kbt_key_cmp);
  if (!kbt->action_index || !kbt->key_index) {
    xine_sarray_delete (kbt->action_index);
    xine_sarray_delete (kbt->key_index);
    free (kbt);
    return NULL;
  }
#if defined (XINE_SARRAY) && (XINE_SARRAY >= 3)
  xine_sarray_set_hash (kbt->action_index, _kbt_action_hash, 64);
  xine_sarray_set_hash (kbt->key_index, _kbt_key_hash, 62);
#endif

  /* Check for space to hold the default table plus a number of user defined aliases.
   * gcc should be smart enough to optimize this away when OK.
   * More likely, it will bail out during "struct kbinding_s { ..." if not. */
  if (KBT_NUM_BASE > KBT_MAX_ENTRIES) {
    fprintf(stderr, "%s(%d):\n"
		    "  Too many entries in default_binding_table[].\n"
		    "  Increase KBT_MAX_ENTRIES to at least %zd.\n",
	    __XINE_FUNCTION__, __LINE__, KBT_NUM_BASE + 100);
    abort();
  }

  kbt->mod_names[0] = _("Ctrl");
  kbt->mod_names[1] = _("Alt");
  kbt->mod_names[2] = _("Mouse");
  kbt->mod_names[3] = _("Super");
  kbt->mod_names[4] = _("AltGr");

  memset (kbt->rev_id, 255, sizeof (kbt->rev_id));
  {
    const default_binding_t *t;
    kbinding_entry_t *n;
    unsigned int u;

    for (u = 0, t = default_binding_table, n = kbt->base; t < default_binding_table + KBT_NUM_BASE; u++, t++, n++) {
      n->name       = t->name.s;
      n->id         = t->id;
      n->index      = u;
      n->flags      = t->flags | KBE_FLAG_default;
      n->comment    = gettext (t->comment);
      kbt->rev_id[n->id & KBT_REV_ID_MASK] = u;
      n->key        = t->key.s;
      n->modifier   = t->modifier;
      _kbt_index_add (kbt, n, 1);
    }
    kbt->num_entries = u;
  }

  if (keymap_file)
    _kbinding_load_config (kbt, keymap_file);

  /* Just to check is there redundant entries, and inform user */
  _kbt_check_redundancy (kbt);

  return kbt;
}

/*
 * Free a keybindings object.
 */
void kbindings_delete (kbinding_t **_kbt) {
  kbinding_t *kbt;
  int nkeys = 0, i;

  if (!_kbt)
    return;
  kbt = *_kbt;
  if (!kbt)
    return;

  kbt->last = NULL;

  for (i = kbt->num_entries - 1; i >= (int)KBT_NUM_BASE; i--) {
    kbinding_entry_t *e = kbt->alias[i - KBT_NUM_BASE];
    if (e) {
      e->name = NULL;
      e->comment = NULL;
      nkeys += refs_unref (&e->key);
      kbt->alias[i - KBT_NUM_BASE] = NULL;
      free (e);
    }
  }
  for (; i >= 0; i--) {
    kbinding_entry_t *e = kbt->base + i;
    e->name = NULL;
    e->comment = NULL;
    nkeys += refs_unref (&e->key);
  }
  if (kbt->gui && (kbt->gui->verbosity >= 2)) {
#if defined (XINE_SARRAY) && (XINE_SARRAY >= 4)
    unsigned int an = xine_sarray_size (kbt->action_index);
    unsigned int aq = xine_sarray_hash_quality (kbt->action_index);
    unsigned int kn = xine_sarray_size (kbt->key_index);
    unsigned int kq = xine_sarray_hash_quality (kbt->key_index);
    printf ("gui.kbindings.delete.hash_quality action (%u, %u.%u%%), key (%u, %u.%u%%).\n",
      an, aq / 10u, aq % 10u, kn, kq / 10u, kq % 10u);
#endif
    if (nkeys)
      printf ("gui.kbindings.delete.keys.free (%d).\n", nkeys);
  }
  xine_sarray_delete (kbt->key_index);
  xine_sarray_delete (kbt->action_index);
  free (kbt);
  *_kbt = NULL;
}

/*
 * Return a duplicated table from kbt.
 */
kbinding_t *kbindings_dup (kbinding_t *kbt) {
  int         i;
  kbinding_t *k;

  ABORT_IF_NULL(kbt);

  k = (kbinding_t *)calloc (1, sizeof (*k));
  if (!k)
    return NULL;

  *k = *kbt;
  k->last = NULL;
  k->action_index = xine_sarray_new (KBT_MAX_ENTRIES, _kbt_action_cmp);
  k->key_index = xine_sarray_new (KBT_MAX_ENTRIES, _kbt_key_cmp);
  if (!k->action_index || !k->key_index) {
    xine_sarray_delete (k->action_index);
    xine_sarray_delete (k->key_index);
    free (k);
    return NULL;
  }
#if defined (XINE_SARRAY) && (XINE_SARRAY >= 3)
  xine_sarray_set_hash (k->action_index, _kbt_action_hash, 64);
  xine_sarray_set_hash (k->key_index, _kbt_key_hash, 62);
#endif

  for (i = 0; i < (int)KBT_NUM_BASE; i++) {
    kbinding_entry_t *n = k->base + i, *o = kbt->base + i;
    n->key = refs_ref (o->key);
    _kbt_index_add (k, n, 1);
  }
  for (; i < kbt->num_entries; i++) {
    kbinding_entry_t *o = kbt->alias[i - KBT_NUM_BASE];
    kbinding_entry_t *n = (kbinding_entry_t *)malloc (sizeof (*n));
    if (!n)
      break;
    *n = *o;
    n->key = refs_ref (o->key);
    _kbt_index_add (k, n, 1);
    k->alias[i - KBT_NUM_BASE] = n;
  }
  k->num_entries = i;

  return k;
}

kbinding_entry_t *kbindings_entry_from_key (kbinding_t *kbt, const char *key, int modifier) {
  kbinding_entry_t *e, e2;
  refs_dummy_t dummy;
  const char *found_key;
  const union {uint8_t s[4]; uint32_t i;} mask = {"\xff\xff\xff\xf0"}, kp_0 = {"KP_0"};
  int i;

  if (!kbt || !key)
    return NULL;

  e2.key = refs_set_dummy (&dummy, key);
  modifier &= (MODIFIER_CTRL | MODIFIER_META | MODIFIER_MOD3 | MODIFIER_MOD4 | MODIFIER_MOD5);
  e2.modifier = modifier;
  e = kbt->last;
  do {
    union {const char *s; const int32_t *v;} v1, v2;
    const union {char s[4]; int32_t v;} last_mask = {{0, 0, 0, 255}};

    if (!e)
      break;
    /* "A" != "a" */
    v1.s = e->key;
    v2.s = e2.key;
    if (*v1.v != *v2.v)
      break;
    do {
      v1.v++, v2.v++;
    } while (*v1.v == *v2.v);
    if ((v1.v[-1] | v2.v[-1]) & last_mask.v)
      break;
    if (e->modifier == e2.modifier)
      return e;
  } while (0);

  i = xine_sarray_binary_search (kbt->key_index, &e2);
  found_key = e2.key;
  refs_unref (&e2.key);
  if (i >= 0) {
    kbt->last = e = xine_sarray_get (kbt->key_index, i);
    return e;
  }
  /* convenience fallback: if key is known without modifiers but not with the
   * supplied ones, assume the former. */
  if (found_key != dummy.s + 4) {
    for (i = (~i) - 1; i >= 0; i--) {
      e = xine_sarray_get (kbt->key_index, i);
      if (e->key != found_key)
        break;
      if (!e->modifier) {
        kbt->last = e;
        return e;
      }
    }
  }
  /* fallback 2: "KP_{0..9}" -> "{0..9}". */
  if ((dummy.i[1] & mask.i) == kp_0.i) {
    e2.key = refs_set_dummy (&dummy, key + 3);
    e2.modifier = modifier;
    i = xine_sarray_binary_search (kbt->key_index, &e2);
    found_key = e2.key;
    refs_unref (&e2.key);
    if (i >= 0) {
      kbt->last = e = xine_sarray_get (kbt->key_index, i);
      return e;
    }
  }
  return NULL;
}

int kbindings_entry_set (kbinding_t *kbt, int index, int modifier, const char *key) {
  const union {char s[4]; int32_t v;} _void = {"void"};
  kbinding_entry_t *e;
  refs_dummy_t dummy;
  int idx = (index < 0) ? ~index : index;

  /* paranoia */
  if (!kbt)
    return -3;
  if (!XITK_0_TO_MAX_MINUS_1 (idx, kbt->num_entries))
    return -3;
  e = _kbt_entry (kbt, idx);
  if (!e)
    return -3;

  if (kbt->gui->verbosity >= 2)
    printf ("gui.kbindings.key.set ([%d] %s, %s, 0x%0x).\n", idx, e->name, key ? key : "NULL", (unsigned int)modifier);
        
  if (!key) {
    /* reset */
    if (index < 0)
      return _kbt_reset (kbt);
    if (XITK_0_TO_MAX_MINUS_1 (idx, KBT_NUM_BASE)) {
      modifier = default_binding_table[idx].modifier;
      _kbt_set_dummy_default (&dummy, default_binding_table + idx);
    } else {
      modifier = 0;
      memcpy (dummy.s, refs_str_void, sizeof (refs_str_void));
    }
  } else {
    refs_set_dummy (&dummy, key);
  }

  modifier &= (MODIFIER_CTRL | MODIFIER_META | MODIFIER_MOD3 | MODIFIER_MOD4 | MODIFIER_MOD5);

  if ((dummy.i[1] | 0x20202020) == _void.v) {
    /* delete */
    if (index < 0)
      return -2;
    if (e->flags & KBE_FLAG_void)
      return -2;
    if (idx >= (int)KBT_NUM_BASE) {
      /* remove alias */
      if (e == kbt->last)
        kbt->last = NULL;
      _kbt_index_remove (kbt, e, 1);
      e->name = NULL;
      e->comment = NULL;
      refs_unref (&e->key);
      free (e);
      kbt->num_entries--;
      for (; idx < kbt->num_entries; idx++) {
        kbt->alias[idx - KBT_NUM_BASE] = kbt->alias[idx - KBT_NUM_BASE + 1];
        kbt->alias[idx - KBT_NUM_BASE]->index = idx;
      }
      return -1;
    } else {
      kbinding_entry_t e2 = { .name = e->name, .flags = KBE_FLAG_alias };
      int ai = xine_sarray_binary_search (kbt->action_index, &e2);
      if (ai >= 0) {
        /* turn alias into new base entry */
        kbinding_entry_t *a = xine_sarray_get (kbt->action_index, ai);
        ai = a->index;
        if ((ai >= (int)KBT_NUM_BASE) && (ai < kbt->num_entries)) {
          if (a == kbt->last)
            kbt->last = NULL;
          _kbt_index_remove (kbt, e, 0);
          _kbt_index_remove (kbt, a, 1);
          e->modifier = a->modifier;
          refs_unref (&e->key);
          e->key = a->key;
          a->name = NULL;
          a->comment = NULL;
          free (a);
          kbt->num_entries--;
          for (; ai < kbt->num_entries; ai++) {
            kbt->alias[ai - KBT_NUM_BASE] = kbt->alias[ai - KBT_NUM_BASE + 1];
            kbt->alias[ai - KBT_NUM_BASE]->index = ai;
          }
          _kbt_index_add (kbt, e, 0);
          _kbt_entry_test_default (e, idx);
          return -1;
        }
      }
      /* keep base entry, just reset */
      _kbt_index_remove (kbt, e, 0);
      refs_unref (&e->key);
      _kbt_entry_set_void (e);
      /* is void now. _kbt_index_add (kbt, e, 0); */
      _kbt_entry_test_default (e, idx);
      return -1;
    }
  } else if (index >= 0) {
    /* change */
    int i = _kbt_change_dummy_key (kbt, e, &dummy, modifier);
    if (i >= 0) {
      /* no change or key already in use. */
      return (i == e->index) ? -2 : i;
    }
    /* new key now set. */
    if (idx < (int)KBT_NUM_BASE)
      _kbt_entry_test_default (e, idx);
    return -1;
  } else {
    kbinding_entry_t *a;
    int i;
    /* add alias */
    if (kbt->num_entries >= KBT_MAX_ENTRIES)
      return -4;
    a = malloc (sizeof (*a));
    if (!a)
      return -3;
    _kbt_entry_set_void (a);
    i = _kbt_change_dummy_key (kbt, a, &dummy, modifier);
    if (i >= 0) {
      /* key already in use */
      free (a);
      return i;
    }
    /* set new alias */
    a->name     = e->name;
    a->id       = e->id;
    a->flags    = (e->flags | KBE_FLAG_alias) & ~KBE_FLAG_default;
    a->comment = e->comment;
    a->index = kbt->num_entries;
    kbt->alias[kbt->num_entries - KBT_NUM_BASE] = a;
    xine_sarray_add (kbt->action_index, a);
    kbt->num_entries++;
    return -1;
  }
}

const kbinding_entry_t *kbindings_entry_from_index (kbinding_t *kbt, int index) {
  if (!kbt)
    return NULL;
  if (!XITK_0_TO_MAX_MINUS_1 (index, kbt->num_entries))
    return NULL;
  return (index < (int)KBT_NUM_BASE) ? kbt->base + index : kbt->alias[index - KBT_NUM_BASE];
}

int kbindings_get_num_entries (kbinding_t *kbt) {
  return kbt ? kbt->num_entries : 0;
}

const char *kbindings_action_name (kbinding_t *kbt, int action) {
  unsigned int u;

  if (!kbt)
    return "???";
  u = (action < 0) ? ~action : action;
  u = kbt->rev_id[u & KBT_REV_ID_MASK];
  if (u >= KBT_NUM_BASE)
    return "???";
  return (action >= 0) ? default_binding_table[u].name.s : gettext (default_binding_table[u].comment);
}

const char * const *kbindings_modifier_names (kbinding_t *kbt) {
  return kbt ? kbt->mod_names : NULL;
}
