/*
 * Copyright (C) 2019-2025 the xine project
 *
 * This file is part of xine, a free video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 * A list of label buttons, optionally with a ... to make all choices available.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <string.h>

#include "_xitk.h"
#include "xitkintl.h"
#include "button_list.h"
#include "labelbutton.h"
#include "skin.h"

#define MAX_BUTTONS 64
/* mrlbrowser calls "show" on the entire widget list.
 * we should not add our hidden ones there, and keep "..." separate.
 * big example: num = 7, max_buttons = 4, visible = 3, (hidden), [shown]
 * 0                 first                   num         last
 * (101) (102) (103) [104] [105] [106] (107) (   ) (   )        [...]
 * medium example: num = 4, max_buttons = 4, visible = 4
 * 0 = first               num = last
 * [101] [102] [103] [104]            (...)
 * small exammple: num = 2, max_buttons = 4, visible = 2
 * 0 = first   num = last
 * [101] [102]            (...) */

typedef struct {
  xitk_widget_t       w;
  xitk_int_callback_t callback;
  widget_event_t      event;
  int                *still_here, select_limit, last_i, num, visible, first, last, dx, dy;
  uint32_t            widget_type, flags;
  xitk_widget_t      *hide, *swap, *widgets[MAX_BUTTONS];
} xitk_button_list_t;

/* optimization: instead of hide/disable, remove hidden widgets from group temporarily. */
static void xitk_button_list_remove (xitk_button_list_t *bl) {
  int i, e;
  /* [101] and [   ] */
  for (i = bl->first, e = xitk_min (i + bl->visible, bl->last); i < e; i++)
    xitk_widget_set_parent (bl->widgets[i], bl->hide);
}

/* optimization: instead of show/enable, add to group again, and paint manually. */
static void xitk_button_list_add (xitk_button_list_t *bl, int more) {
  int b = bl->first, e = xitk_min (b + bl->visible, bl->last), i, x = bl->event.x, y = bl->event.y;

  bl->event.type = (bl->w.wl->flags & XITK_WL_EXPOSED) ? WIDGET_EVENT_PAINT : WIDGET_EVENT_NONE;
  
  /* [101] [   } */
  for (i = b; i < e; i++) {
    xitk_widget_set_parent (bl->widgets[i], &bl->w);
    xitk_set_widget_pos (bl->widgets[i], bl->event.x, bl->event.y);
    bl->widgets[i]->event (bl->widgets[i], &bl->event);
    bl->event.x += bl->dx;
    bl->event.y += bl->dy;
  }
  /* [...] */
  if (more) {
    if (bl->num > bl->visible) {
      xitk_widget_set_parent (bl->swap, &bl->w);
      xitk_set_widget_pos (bl->swap, bl->event.x, bl->event.y);
      if (bl->swap)
        bl->swap->event (bl->swap, &bl->event);
    } else {
      xitk_widget_set_parent (bl->swap, bl->hide);
    }
  }
  bl->event.x = x;
  bl->event.y = y;
}

static void xitk_button_list_click (xitk_widget_t *w, void *data, int state, unsigned int modifier) {
  xitk_button_list_t *bl = (xitk_button_list_t *)data;
  xitk_widget_t *last_w = (state && (bl->select_limit > 0)) ? ((bl->last_i >= 0) ? bl->widgets[bl->last_i] : NULL) : w;

  bl->last_i = xitk_widget_user_id (w);
  if (last_w != w)
    xitk_widgets_state (&last_w, 1, XITK_WIDGET_STATE_ON, 0);
  if (bl->callback) {
    int still_here = 1;

    bl->still_here = &still_here;
    bl->callback (w, bl->w.userdata, state, modifier);
    if (!still_here)
      return;
    bl->still_here = &bl->dx;
    if (last_w != w)
      bl->callback (last_w, bl->w.userdata, 0, modifier);
  }
}

static void xitk_button_list_swap (xitk_widget_t *w, void *ip, int state, unsigned int modifier) {
  xitk_button_list_t *bl = (xitk_button_list_t *)ip;

  (void)w;
  (void)state;
  (void)modifier;

  xitk_button_list_remove (bl);

  bl->first += bl->visible;
  if (bl->first >= bl->num)
    bl->first = 0;

  xitk_button_list_add (bl, 0);
}

static void xitk_button_list_hull (xitk_button_list_t *bl, const xitk_skin_element_info_t *info) {
  XITK_HV_INIT;
  int d, n = bl->num > bl->visible ? bl->visible + 1 : bl->num;

  if (info) {
    XITK_HV_H (bl->w.pos) = bl->event.x = info->x;
    XITK_HV_V (bl->w.pos) = bl->event.y = info->y;
    d = info->type;
  } else {
    bl->w.pos.w = bl->event.x = bl->event.y = 0;
    d = DIRECTION_LEFT;
  }

  /* bl exists, so bl->widgets[0] exists, too. */
  bl->w.size.w = bl->widgets[0]->size.w;
  bl->event.width = XITK_HV_H (bl->widgets[0]->size);
  bl->event.height = XITK_HV_V (bl->widgets[0]->size);

  switch (d) {
    case DIRECTION_UP:
      XITK_HV_V (bl->w.pos) = bl->event.y - (n - 1) * (bl->event.height + 1);
      bl->dy = -(bl->event.height + 1);
      bl->dx = 0;
      XITK_HV_V (bl->w.size) = n * (bl->event.height + 1) - 1;
      break;
    case DIRECTION_DOWN:
      bl->dy = bl->event.height + 1;
      bl->dx = 0;
      XITK_HV_V (bl->w.size) = n * (bl->event.height + 1) - 1;
      break;
    case DIRECTION_LEFT:
    default:
      XITK_HV_H (bl->w.pos) = bl->event.x - (n - 1) * (bl->event.width + 1);
      bl->dx = -(bl->event.width + 1);
      bl->dy = 0;
      XITK_HV_H (bl->w.size) = n * (bl->event.width + 1) - 1;
      break;
    case DIRECTION_RIGHT:
      bl->dx = bl->event.width + 1;
      bl->dy = 0;
      XITK_HV_H (bl->w.size) = n * (bl->event.width + 1) - 1;
      break;
  }

  bl->first = 0;
  xitk_button_list_add (bl, 1);
}

static void xitk_button_list_new_skin (xitk_button_list_t *bl, const widget_event_t *event) {
  const xitk_skin_element_info_t *info;
  xitk_widget_t *w;
  int i, max;

  /* relay new skin to hidden ones */
  i = 0;
  xitk_container (w, bl->hide->parent.list.head.next, parent.node);
  while (w->parent.node.next) {
    w->state &= ~(XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE);
    w->event (w, event);
    w->state |= XITK_WIDGET_STATE_VISIBLE | XITK_WIDGET_STATE_ENABLE;
    i++;
    xitk_container (w, w->parent.node.next, parent.node);
  }
  if (bl->w.wl->xitk->verbosity >= 2)
    printf ("xitk.button_list.skin.change.hidden (%s) = %d.\n", bl->w.wl->name, i);
  for (i = bl->num; i < bl->last; i++)
    bl->widgets[i]->state &= ~XITK_WIDGET_STATE_ENABLE;

  xitk_button_list_remove (bl);

  info = xitk_skin_get_info (event->skonfig, bl->w.skin_element_name);
  max = info ? info->size : 0;
  if (max <= 0)
    max = 10000;
  if (bl->num > max) {
    max -= 1;
  } else {
    max = bl->num;
  }
  bl->visible = max;

  max = (bl->num + bl->visible - 1) / bl->visible * bl->visible;
  if (max > MAX_BUTTONS)
    max = MAX_BUTTONS;
  if (max > bl->last) {
    /* more "   " */
    xitk_labelbutton_widget_t lb = {
      .nw = {
        .wl = bl->w.wl,
        .group = bl->hide,
        .skin_element_name = bl->w.skin_element_name, /* needed for possible skin change. */
        .add_state = XITK_WIDGET_STATE_VISIBLE,
        .mode_mask = bl->widget_type,
        .mode_value = bl->widget_type
      },
      .button_type    = CLICK_BUTTON,
      .align          = ALIGN_DEFAULT,
      .label          = "",
    };
    const xitk_skin_element_info_t *pinfo = xitk_skin_get_info (event->skonfig, bl->w.skin_element_name);

    for (i = bl->last; i < max; i++) {
      /* avoid multiple identical xitk_skin_get_info () inside xitk_labelbutton_create (). */
      bl->widgets[i] = xitk_info_labelbutton_create (&lb, pinfo);
      if (!bl->widgets[i])
        break;
    }
    bl->last = i;
  } else if (max < bl->last) {
    /* less "   " */
    xitk_widgets_delete (bl->widgets + max, bl->last - max);
    bl->last = max;
  }

  xitk_button_list_hull (bl, info);

  bl->flags |= XITK_WIDGET_STATE_ENABLE;
}

static int xitk_button_list_event (xitk_widget_t *w, const widget_event_t *event) {
  xitk_button_list_t *bl = (xitk_button_list_t *)w;

  if (event && bl && ((bl->w.type & WIDGET_TYPE_MASK) == WIDGET_TYPE_BUTTON_LIST)) {
    switch (event->type) {
      case WIDGET_EVENT_CHANGE_SKIN:
        xitk_button_list_new_skin (bl, event);
        break;
      case WIDGET_EVENT_DESTROY:
        *bl->still_here = 0;
        xitk_widgets_delete (bl->widgets, bl->last);
        xitk_widgets_delete (&bl->swap, 1);
        free (bl->hide);
        break;
      case WIDGET_EVENT_ENABLE:
        if ((bl->w.state ^ bl->flags) & XITK_WIDGET_STATE_ENABLE) {
          int e = xitk_min (bl->first + bl->visible, bl->num);
          bl->flags ^= XITK_WIDGET_STATE_ENABLE;
          xitk_widgets_state (bl->widgets + bl->first, e - bl->first, XITK_WIDGET_STATE_ENABLE, bl->flags);
          if (bl->num > bl->visible)
            xitk_widgets_state (&bl->swap, 1, XITK_WIDGET_STATE_ENABLE, bl->flags);
        }
        break;
      case WIDGET_EVENT_TIPS_TIMEOUT:
        xitk_set_widget_tips_and_timeout (bl->swap, bl->w.tips_string, event->tips_timeout);
        break;
      default: ;
    }
  }
  return 0;
}

xitk_widget_t *xitk_button_list_new (const xitk_button_list_widget_t *nbl, xitk_skin_config_t *skin_config) {
  xitk_button_list_t *bl;
  const xitk_skin_element_info_t *info;
  int i, max;

  if (!nbl)
    return NULL;
  if (!nbl->names)
    return NULL;
  for (i = 0; nbl->names[i]; i++) ;
  if (i == 0)
    return NULL;
  if (i > MAX_BUTTONS)
    i = MAX_BUTTONS;

  bl = (xitk_button_list_t *)xitk_widget_new (&nbl->nw, sizeof (*bl));
  if (!bl)
    return NULL;
  {
    xitk_new_widget_t nw = {
      .wl = bl->w.wl,
      .userdata = bl,
    };
    bl->hide = xitk_widget_new (&nw, 0);
  }
  if (!bl->hide) {
    free (bl);
    return NULL;
  }

  bl->flags = XITK_WIDGET_STATE_ENABLE;

  bl->widget_type = nbl->widget_type_flags | WIDGET_GROUP_MEMBER | WIDGET_GROUP_BUTTON_LIST;
  bl->callback = nbl->callback;
  bl->still_here = &bl->dx;
  bl->select_limit = nbl->select_limit;
  bl->last_i = -1;

  info = xitk_skin_get_info (skin_config, bl->w.skin_element_name);
  max = info ? info->size : 0;
  if (max <= 0)
    max = 10000;
  if (max == 1) {
    i = 1;
  } else if (i > max) {
    max -= 1;
  } else {
    max = i;
  }
  bl->num = i;
  bl->visible = max;

  bl->last = (bl->num + bl->visible - 1) / bl->visible * bl->visible;
  if (bl->last > MAX_BUTTONS)
    bl->last = MAX_BUTTONS;

  /* bl->first = 0; already done by xitk_widget_new (). */
  bl->w.type = WIDGET_GROUP | WIDGET_TYPE_BUTTON_LIST;
  xitk_widget_state_from_info (&bl->w, info);
  bl->w.event = xitk_button_list_event;

  {
    xitk_labelbutton_widget_t lb = {
      .nw = {
        .wl = bl->w.wl,
        .userdata = bl,
        .group = bl->hide,
        .skin_element_name = bl->w.skin_element_name, /* needed for possible skin change. */
        .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE,
        .mode_mask = bl->widget_type,
        .mode_value = bl->widget_type
      },
      .button_type = (bl->select_limit > 0) ? RADIO_BUTTON : CLICK_BUTTON,
      .align       = ALIGN_DEFAULT,
      .callback    = xitk_button_list_click
    };
    const xitk_skin_element_info_t *pinfo = xitk_skin_get_info (skin_config, bl->w.skin_element_name);
    int n = bl->num > 0 ? bl->num : 1;

    /* "101" */
    for (i = 0; i < n; i++) {
      lb.label       = nbl->names[i];
      lb.nw.tips     = nbl->tips[i];
      lb.nw.user_id  = i;
      /* avoid multiple identical xitk_skin_get_info () inside xitk_labelbutton_create (). */
      bl->widgets[i] = xitk_info_labelbutton_create (&lb, pinfo);
      if (!bl->widgets[i])
        break;
    }
    if (!i) {
      free (bl);
      return NULL;
    }
    bl->num = i;

    /* "   " */
    lb.nw.add_state = XITK_WIDGET_STATE_VISIBLE;
    lb.nw.userdata = NULL;
    lb.button_type = CLICK_BUTTON;
    lb.label = "";
    lb.nw.tips = NULL;
    lb.callback = NULL;
    for (; i < bl->last; i++) {
      bl->widgets[i] = xitk_info_labelbutton_create (&lb, pinfo);
      if (!bl->widgets[i])
        break;
    }
    bl->last = i;

    /* "..." */
    lb.nw.add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE;
    lb.nw.userdata = bl;
    lb.callback = xitk_button_list_swap;
    lb.label    = _("...");
    lb.nw.tips  = nbl->nw.tips;
    bl->swap    = xitk_info_labelbutton_create (&lb, info);
  }

  xitk_button_list_hull (bl, info);

  return _xitk_new_widget_apply (&nbl->nw, &bl->w);
}

xitk_widget_t *xitk_button_list_find (xitk_widget_t *w, const char *name) {
  xitk_button_list_t *bl = (xitk_button_list_t *)w;
  int i;
  if (!bl || !name)
    return NULL;
  if ((bl->w.type & WIDGET_TYPE_MASK) != WIDGET_TYPE_BUTTON_LIST)
    return NULL;
  for (i = 0; i < bl->num; i++) {
    const char *p = xitk_labelbutton_get_label (bl->widgets[i]);
    if (p && !strcasecmp (p, name)) {
      bl->last_i = i;
      if (bl->select_limit > 0)
        xitk_widgets_state (bl->widgets + i, 1, XITK_WIDGET_STATE_ON, ~0u);
      return bl->widgets[i];
    }
  }
  return NULL;
}
