/* SPDX-License-Identifier: GPL-2.0-or-later */
/*
 * Afatech AF9033 demodulator driver
 *
 * Copyright (C) 2009 Antti Palosaari <crope@iki.fi>
 * Copyright (C) 2012 Antti Palosaari <crope@iki.fi>
 */

#ifndef AF9033_PRIV_H
#define AF9033_PRIV_H

#include <media/dvb_frontend.h>
#include "af9033.h"
#include <linux/math64.h>
#include <linux/regmap.h>
#include <linux/kernel.h>
#include <linux/int_log.h>

struct reg_val {
	u32 reg;
	u8  val;
};

struct reg_val_mask {
	u32 reg;
	u8  val;
	u8  mask;
};

struct coeff {
	u32 clock;
	u32 bandwidth_hz;
	u8 val[36];
};

struct clock_adc {
	u32 clock;
	u32 adc;
};

struct val_snr {
	u32 val;
	u8 snr;
};

/* Xtal clock vs. ADC clock lookup table */
static const struct clock_adc clock_adc_lut[] = {
	{ 16384000, 20480000 },
	{ 20480000, 20480000 },
	{ 36000000, 20250000 },
	{ 30000000, 20156250 },
	{ 26000000, 20583333 },
	{ 28000000, 20416667 },
	{ 32000000, 20500000 },
	{ 34000000, 20187500 },
	{ 24000000, 20500000 },
	{ 22000000, 20625000 },
	{ 12000000, 20250000 },
};

/* Pre-calculated coeff lookup table */
static const struct coeff coeff_lut[] = {
	/* 12.000 MHz */
	{ 12000000, 8000000, {
		0x01, 0xce, 0x55, 0xc9, 0x00, 0xe7, 0x2a, 0xe4, 0x00, 0x73,
		0x99, 0x0f, 0x00, 0x73, 0x95, 0x72, 0x00, 0x73, 0x91, 0xd5,
		0x00, 0x39, 0xca, 0xb9, 0x00, 0xe7, 0x2a, 0xe4, 0x00, 0x73,
		0x95, 0x72, 0x37, 0x02, 0xce, 0x01 }
	},
	{ 12000000, 7000000, {
		0x01, 0x94, 0x8b, 0x10, 0x00, 0xca, 0x45, 0x88, 0x00, 0x65,
		0x25, 0xed, 0x00, 0x65, 0x22, 0xc4, 0x00, 0x65, 0x1f, 0x9b,
		0x00, 0x32, 0x91, 0x62, 0x00, 0xca, 0x45, 0x88, 0x00, 0x65,
		0x22, 0xc4, 0x88, 0x02, 0x95, 0x01 }
	},
	{ 12000000, 6000000, {
		0x01, 0x5a, 0xc0, 0x56, 0x00, 0xad, 0x60, 0x2b, 0x00, 0x56,
		0xb2, 0xcb, 0x00, 0x56, 0xb0, 0x15, 0x00, 0x56, 0xad, 0x60,
		0x00, 0x2b, 0x58, 0x0b, 0x00, 0xad, 0x60, 0x2b, 0x00, 0x56,
		0xb0, 0x15, 0xf4, 0x02, 0x5b, 0x01 }
	},
};

/*
 * Afatech AF9033 demod init
 */
static const struct reg_val ofsm_init[] = {
	{ 0x800051, 0x01 },
	{ 0x800070, 0x0a },
	{ 0x80007e, 0x04 },
	{ 0x800081, 0x0a },
	{ 0x80008a, 0x01 },
	{ 0x80008e, 0x01 },
	{ 0x800092, 0x06 },
	{ 0x800099, 0x01 },
	{ 0x80009f, 0xe1 },
	{ 0x8000a0, 0xcf },
	{ 0x8000a3, 0x01 },
	{ 0x8000a5, 0x01 },
	{ 0x8000a6, 0x01 },
	{ 0x8000a9, 0x00 },
	{ 0x8000aa, 0x01 },
	{ 0x8000b0, 0x01 },
	{ 0x8000c4, 0x05 },
	{ 0x8000c8, 0x19 },
	{ 0x80f000, 0x0f },
	{ 0x80f016, 0x10 },
	{ 0x80f017, 0x04 },
	{ 0x80f018, 0x05 },
	{ 0x80f019, 0x04 },
	{ 0x80f01a, 0x05 },
	{ 0x80f021, 0x03 },
	{ 0x80f022, 0x0a },
	{ 0x80f023, 0x0a },
	{ 0x80f02b, 0x00 },
	{ 0x80f02c, 0x01 },
	{ 0x80f064, 0x03 },
	{ 0x80f065, 0xf9 },
	{ 0x80f066, 0x03 },
	{ 0x80f067, 0x01 },
	{ 0x80f06f, 0xe0 },
	{ 0x80f070, 0x03 },
	{ 0x80f072, 0x0f },
	{ 0x80f073, 0x03 },
	{ 0x80f078, 0x00 },
	{ 0x80f087, 0x00 },
	{ 0x80f09b, 0x3f },
	{ 0x80f09c, 0x00 },
	{ 0x80f09d, 0x20 },
	{ 0x80f09e, 0x00 },
	{ 0x80f09f, 0x0c },
	{ 0x80f0a0, 0x00 },
	{ 0x80f130, 0x04 },
	{ 0x80f132, 0x04 },
	{ 0x80f144, 0x1a },
	{ 0x80f146, 0x00 },
	{ 0x80f14a, 0x01 },
	{ 0x80f14c, 0x00 },
	{ 0x80f14d, 0x00 },
	{ 0x80f14f, 0x04 },
	{ 0x80f158, 0x7f },
	{ 0x80f15a, 0x00 },
	{ 0x80f15b, 0x08 },
	{ 0x80f15d, 0x03 },
	{ 0x80f15e, 0x05 },
	{ 0x80f163, 0x05 },
	{ 0x80f166, 0x01 },
	{ 0x80f167, 0x40 },
	{ 0x80f168, 0x0f },
	{ 0x80f17a, 0x00 },
	{ 0x80f17b, 0x00 },
	{ 0x80f183, 0x01 },
	{ 0x80f19d, 0x40 },
	{ 0x80f1bc, 0x36 },
	{ 0x80f1bd, 0x00 },
	{ 0x80f1cb, 0xa0 },
	{ 0x80f1cc, 0x01 },
	{ 0x80f204, 0x10 },
	{ 0x80f214, 0x00 },
	{ 0x80f40e, 0x0a },
	{ 0x80f40f, 0x40 },
	{ 0x80f410, 0x08 },
	{ 0x80f55f, 0x0a },
	{ 0x80f561, 0x15 },
	{ 0x80f562, 0x20 },
	{ 0x80f5df, 0xfb },
	{ 0x80f5e0, 0x00 },
	{ 0x80f5e3, 0x09 },
	{ 0x80f5e4, 0x01 },
	{ 0x80f5e5, 0x01 },
	{ 0x80f5f8, 0x01 },
	{ 0x80f5fd, 0x01 },
	{ 0x80f600, 0x05 },
	{ 0x80f601, 0x08 },
	{ 0x80f602, 0x0b },
	{ 0x80f603, 0x0e },
	{ 0x80f604, 0x11 },
	{ 0x80f605, 0x14 },
	{ 0x80f606, 0x17 },
	{ 0x80f607, 0x1f },
	{ 0x80f60e, 0x00 },
	{ 0x80f60f, 0x04 },
	{ 0x80f610, 0x32 },
	{ 0x80f611, 0x10 },
	{ 0x80f707, 0xfc },
	{ 0x80f708, 0x00 },
	{ 0x80f709, 0x37 },
	{ 0x80f70a, 0x00 },
	{ 0x80f78b, 0x01 },
	{ 0x80f80f, 0x40 },
	{ 0x80f810, 0x54 },
	{ 0x80f811, 0x5a },
	{ 0x80f905, 0x01 },
	{ 0x80fb06, 0x03 },
	{ 0x80fd8b, 0x00 },
};

/*
 * Infineon TUA 9001 tuner init
 * AF9033_TUNER_TUA9001    = 0x27
 */
static const struct reg_val tuner_init_tua9001[] = {
	{ 0x800046, 0x27 },
	{ 0x800057, 0x00 },
	{ 0x800058, 0x01 },
	{ 0x80005f, 0x00 },
	{ 0x800060, 0x00 },
	{ 0x80006d, 0x00 },
	{ 0x800071, 0x05 },
	{ 0x800072, 0x02 },
	{ 0x800074, 0x01 },
	{ 0x800075, 0x03 },
	{ 0x800076, 0x02 },
	{ 0x800077, 0x00 },
	{ 0x800078, 0x01 },
	{ 0x800079, 0x00 },
	{ 0x80007a, 0x7e },
	{ 0x80007b, 0x3e },
	{ 0x800093, 0x00 },
	{ 0x800094, 0x01 },
	{ 0x800095, 0x02 },
	{ 0x800096, 0x01 },
	{ 0x800098, 0x0a },
	{ 0x80009b, 0x05 },
	{ 0x80009c, 0x80 },
	{ 0x8000b3, 0x00 },
	{ 0x8000c5, 0x01 },
	{ 0x8000c6, 0x00 },
	{ 0x8000c9, 0x5d },
	{ 0x80f007, 0x00 },
	{ 0x80f01f, 0x82 },
	{ 0x80f020, 0x00 },
	{ 0x80f029, 0x82 },
	{ 0x80f02a, 0x00 },
	{ 0x80f047, 0x00 },
	{ 0x80f054, 0x00 },
	{ 0x80f055, 0x00 },
	{ 0x80f077, 0x01 },
	{ 0x80f1e6, 0x00 },
};

/*
 * Fitipower FC0011 tuner init
 * AF9033_TUNER_FC0011    = 0x28
 */
static const struct reg_val tuner_init_fc0011[] = {
	{ 0x800046, 0x28 },
	{ 0x800057, 0x00 },
	{ 0x800058, 0x01 },
	{ 0x80005f, 0x00 },
	{ 0x800060, 0x00 },
	{ 0x800068, 0xa5 },
	{ 0x80006e, 0x01 },
	{ 0x800071, 0x0a },
	{ 0x800072, 0x02 },
	{ 0x800074, 0x01 },
	{ 0x800079, 0x01 },
	{ 0x800093, 0x00 },
	{ 0x800094, 0x00 },
	{ 0x800095, 0x00 },
	{ 0x800096, 0x00 },
	{ 0x80009b, 0x2d },
	{ 0x80009c, 0x60 },
	{ 0x80009d, 0x23 },
	{ 0x8000a4, 0x50 },
	{ 0x8000ad, 0x50 },
	{ 0x8000b3, 0x01 },
	{ 0x8000b7, 0x88 },
	{ 0x8000b8, 0xa6 },
	{ 0x8000c5, 0x01 },
	{ 0x8000c6, 0x01 },
	{ 0x8000c9, 0x69 },
	{ 0x80f007, 0x00 },
	{ 0x80f00a, 0x1b },
	{ 0x80f00b, 0x1b },
	{ 0x80f00c, 0x1b },
	{ 0x80f00d, 0x1b },
	{ 0x80f00e, 0xff },
	{ 0x80f00f, 0x01 },
	{ 0x80f010, 0x00 },
	{ 0x80f011, 0x02 },
	{ 0x80f012, 0xff },
	{ 0x80f013, 0x01 },
	{ 0x80f014, 0x00 },
	{ 0x80f015, 0x02 },
	{ 0x80f01b, 0xef },
	{ 0x80f01c, 0x01 },
	{ 0x80f01d, 0x0f },
	{ 0x80f01e, 0x02 },
	{ 0x80f01f, 0x6e },
	{ 0x80f020, 0x00 },
	{ 0x80f025, 0xde },
	{ 0x80f026, 0x00 },
	{ 0x80f027, 0x0a },
	{ 0x80f028, 0x03 },
	{ 0x80f029, 0x6e },
	{ 0x80f02a, 0x00 },
	{ 0x80f047, 0x00 },
	{ 0x80f054, 0x00 },
	{ 0x80f055, 0x00 },
	{ 0x80f077, 0x01 },
	{ 0x80f1e6, 0x00 },
};

/*
 * Fitipower FC0012 tuner init
 * AF9033_TUNER_FC0012    = 0x2e
 */
static const struct reg_val tuner_init_fc0012[] = {
	{ 0x800046, 0x2e },
	{ 0x800057, 0x00 },
	{ 0x800058, 0x01 },
	{ 0x800059, 0x01 },
	{ 0x80005f, 0x00 },
	{ 0x800060, 0x00 },
	{ 0x80006d, 0x00 },
	{ 0x800071, 0x05 },
	{ 0x800072, 0x02 },
	{ 0x800074, 0x01 },
	{ 0x800075, 0x03 },
	{ 0x800076, 0x02 },
	{ 0x800077, 0x01 },
	{ 0x800078, 0x00 },
	{ 0x800079, 0x00 },
	{ 0x80007a, 0x90 },
	{ 0x80007b, 0x90 },
	{ 0x800093, 0x00 },
	{ 0x800094, 0x01 },
	{ 0x800095, 0x02 },
	{ 0x800096, 0x01 },
	{ 0x800098, 0x0a },
	{ 0x80009b, 0x05 },
	{ 0x80009c, 0x80 },
	{ 0x8000b3, 0x00 },
	{ 0x8000c5, 0x01 },
	{ 0x8000c6, 0x00 },
	{ 0x8000c9, 0x5d },
	{ 0x80f007, 0x00 },
	{ 0x80f01f, 0xa0 },
	{ 0x80f020, 0x00 },
	{ 0x80f029, 0x82 },
	{ 0x80f02a, 0x00 },
	{ 0x80f047, 0x00 },
	{ 0x80f054, 0x00 },
	{ 0x80f055, 0x00 },
	{ 0x80f077, 0x01 },
	{ 0x80f1e6, 0x00 },
};

/*
 * MaxLinear MxL5007T tuner init
 * AF9033_TUNER_MXL5007T    = 0xa0
 */
static const struct reg_val tuner_init_mxl5007t[] = {
	{ 0x800046, 0x1b },
	{ 0x800057, 0x01 },
	{ 0x800058, 0x01 },
	{ 0x80005f, 0x00 },
	{ 0x800060, 0x00 },
	{ 0x800068, 0x96 },
	{ 0x800071, 0x05 },
	{ 0x800072, 0x02 },
	{ 0x800074, 0x01 },
	{ 0x800079, 0x01 },
	{ 0x800093, 0x00 },
	{ 0x800094, 0x00 },
	{ 0x800095, 0x00 },
	{ 0x800096, 0x00 },
	{ 0x8000b3, 0x01 },
	{ 0x8000c1, 0x01 },
	{ 0x8000c2, 0x00 },
	{ 0x80f007, 0x00 },
	{ 0x80f00c, 0x19 },
	{ 0x80f00d, 0x1a },
	{ 0x80f012, 0xda },
	{ 0x80f013, 0x00 },
	{ 0x80f014, 0x00 },
	{ 0x80f015, 0x02 },
	{ 0x80f01f, 0x82 },
	{ 0x80f020, 0x00 },
	{ 0x80f029, 0x82 },
	{ 0x80f02a, 0x00 },
	{ 0x80f077, 0x02 },
	{ 0x80f1e6, 0x00 },
};

/*
 * NXP TDA18218HN tuner init
 * AF9033_TUNER_TDA18218    = 0xa1
 */
static const struct reg_val tuner_init_tda18218[] = {
	{0x800046, 0xa1},
	{0x800057, 0x01},
	{0x800058, 0x01},
	{0x80005f, 0x00},
	{0x800060, 0x00},
	{0x800071, 0x05},
	{0x800072, 0x02},
	{0x800074, 0x01},
	{0x800079, 0x01},
	{0x800093, 0x00},
	{0x800094, 0x00},
	{0x800095, 0x00},
	{0x800096, 0x00},
	{0x8000b3, 0x01},
	{0x8000c3, 0x01},
	{0x8000c4, 0x00},
	{0x80f007, 0x00},
	{0x80f00c, 0x19},
	{0x80f00d, 0x1a},
	{0x80f012, 0xda},
	{0x80f013, 0x00},
	{0x80f014, 0x00},
	{0x80f015, 0x02},
	{0x80f01f, 0x82},
	{0x80f020, 0x00},
	{0x80f029, 0x82},
	{0x80f02a, 0x00},
	{0x80f077, 0x02},
	{0x80f1e6, 0x00},
};

/*
 * FCI FC2580 tuner init
 * AF9033_TUNER_FC2580      = 0x32
 */
static const struct reg_val tuner_init_fc2580[] = {
	{ 0x800046, 0x32 },
	{ 0x800057, 0x01 },
	{ 0x800058, 0x00 },
	{ 0x80005f, 0x00 },
	{ 0x800060, 0x00 },
	{ 0x800071, 0x05 },
	{ 0x800072, 0x02 },
	{ 0x800074, 0x01 },
	{ 0x800079, 0x01 },
	{ 0x800093, 0x00 },
	{ 0x800094, 0x00 },
	{ 0x800095, 0x00 },
	{ 0x800096, 0x05 },
	{ 0x8000b3, 0x01 },
	{ 0x8000c5, 0x01 },
	{ 0x8000c6, 0x00 },
	{ 0x8000d1, 0x01 },
	{ 0x80f007, 0x00 },
	{ 0x80f00c, 0x19 },
	{ 0x80f00d, 0x1a },
	{ 0x80f00e, 0x00 },
	{ 0x80f00f, 0x02 },
	{ 0x80f010, 0x00 },
	{ 0x80f011, 0x02 },
	{ 0x80f012, 0x00 },
	{ 0x80f013, 0x02 },
	{ 0x80f014, 0x00 },
	{ 0x80f015, 0x02 },
	{ 0x80f01f, 0x96 },
	{ 0x80f020, 0x00 },
	{ 0x80f029, 0x96 },
	{ 0x80f02a, 0x00 },
	{ 0x80f077, 0x01 },
	{ 0x80f1e6, 0x01 },
};

/*
 * IT9133 AX demod init
 */
static const struct reg_val ofsm_init_it9135_v1[] = {
	{ 0x800051, 0x01 },
	{ 0x800070, 0x0a },
	{ 0x80007e, 0x04 },
	{ 0x800081, 0x0a },
	{ 0x80008a, 0x01 },
	{ 0x80008e, 0x01 },
	{ 0x800092, 0x06 },
	{ 0x800099, 0x01 },
	{ 0x80009f, 0xe1 },
	{ 0x8000a0, 0xcf },
	{ 0x8000a3, 0x01 },
	{ 0x8000a5, 0x01 },
	{ 0x8000a6, 0x01 },
	{ 0x8000a9, 0x00 },
	{ 0x8000aa, 0x01 },
	{ 0x8000b0, 0x01 },
	{ 0x8000c2, 0x05 },
	{ 0x8000c6, 0x19 },
	{ 0x80f000, 0x0f },
	{ 0x80f016, 0x10 },
	{ 0x80f017, 0x04 },
	{ 0x80f018, 0x05 },
	{ 0x80f019, 0x04 },
	{ 0x80f01a, 0x05 },
	{ 0x80f021, 0x03 },
	{ 0x80f022, 0x0a },
	{ 0x80f023, 0x0a },
	{ 0x80f02b, 0x00 },
	{ 0x80f02c, 0x01 },
	{ 0x80f064, 0x03 },
	{ 0x80f065, 0xf9 },
	{ 0x80f066, 0x03 },
	{ 0x80f067, 0x01 },
	{ 0x80f06f, 0xe0 },
	{ 0x80f070, 0x03 },
	{ 0x80f072, 0x0f },
	{ 0x80f073, 0x03 },
	{ 0x80f078, 0x00 },
	{ 0x80f087, 0x00 },
	{ 0x80f09b, 0x3f },
	{ 0x80f09c, 0x00 },
	{ 0x80f09d, 0x20 },
	{ 0x80f09e, 0x00 },
	{ 0x80f09f, 0x0c },
	{ 0x80f0a0, 0x00 },
	{ 0x80f130, 0x04 },
	{ 0x80f132, 0x04 },
	{ 0x80f144, 0x1a },
	{ 0x80f146, 0x00 },
	{ 0x80f14a, 0x01 },
	{ 0x80f14c, 0x00 },
	{ 0x80f14d, 0x00 },
	{ 0x80f14f, 0x04 },
	{ 0x80f158, 0x7f },
	{ 0x80f15a, 0x00 },
	{ 0x80f15b, 0x08 },
	{ 0x80f15d, 0x03 },
	{ 0x80f15e, 0x05 },
	{ 0x80f163, 0x05 },
	{ 0x80f166, 0x01 },
	{ 0x80f167, 0x40 },
	{ 0x80f168, 0x0f },
	{ 0x80f17a, 0x00 },
	{ 0x80f17b, 0x00 },
	{ 0x80f183, 0x01 },
	{ 0x80f19d, 0x40 },
	{ 0x80f1bc, 0x36 },
	{ 0x80f1bd, 0x00 },
	{ 0x80f1cb, 0xa0 },
	{ 0x80f1cc, 0x01 },
	{ 0x80f204, 0x10 },
	{ 0x80f214, 0x00 },
	{ 0x80f40e, 0x0a },
	{ 0x80f40f, 0x40 },
	{ 0x80f410, 0x08 },
	{ 0x80f55f, 0x0a },
	{ 0x80f561, 0x15 },
	{ 0x80f562, 0x20 },
	{ 0x80f5df, 0xfb },
	{ 0x80f5e0, 0x00 },
	{ 0x80f5e3, 0x09 },
	{ 0x80f5e4, 0x01 },
	{ 0x80f5e5, 0x01 },
	{ 0x80f5f8, 0x01 },
	{ 0x80f5fd, 0x01 },
	{ 0x80f600, 0x05 },
	{ 0x80f601, 0x08 },
	{ 0x80f602, 0x0b },
	{ 0x80f603, 0x0e },
	{ 0x80f604, 0x11 },
	{ 0x80f605, 0x14 },
	{ 0x80f606, 0x17 },
	{ 0x80f607, 0x1f },
	{ 0x80f60e, 0x00 },
	{ 0x80f60f, 0x04 },
	{ 0x80f610, 0x32 },
	{ 0x80f611, 0x10 },
	{ 0x80f707, 0xfc },
	{ 0x80f708, 0x00 },
	{ 0x80f709, 0x37 },
	{ 0x80f70a, 0x00 },
	{ 0x80f78b, 0x01 },
	{ 0x80f80f, 0x40 },
	{ 0x80f810, 0x54 },
	{ 0x80f811, 0x5a },
	{ 0x80f905, 0x01 },
	{ 0x80fb06, 0x03 },
	{ 0x80fd8b, 0x00 },
};

/*
 * ITE Tech IT9133 AX Omega tuner init
 * AF9033_TUNER_IT9135_38   = 0x38
 */
static const struct reg_val tuner_init_it9135_38[] = {
	{ 0x800043, 0x00 },
	{ 0x800046, 0x38 },
	{ 0x800051, 0x01 },
	{ 0x80005f, 0x00 },
	{ 0x800060, 0x00 },
	{ 0x800068, 0x0a },
	{ 0x800070, 0x0a },
	{ 0x800071, 0x05 },
	{ 0x800072, 0x02 },
	{ 0x800075, 0x8c },
	{ 0x800076, 0x8c },
	{ 0x800077, 0x8c },
	{ 0x800078, 0xc8 },
	{ 0x800079, 0x01 },
	{ 0x80007e, 0x04 },
	{ 0x80007f, 0x00 },
	{ 0x800081, 0x0a },
	{ 0x800082, 0x12 },
	{ 0x800083, 0x02 },
	{ 0x800084, 0x0a },
	{ 0x800085, 0x03 },
	{ 0x800086, 0xc8 },
	{ 0x800087, 0xb8 },
	{ 0x800088, 0xd0 },
	{ 0x800089, 0xc3 },
	{ 0x80008a, 0x01 },
	{ 0x80008e, 0x01 },
	{ 0x800092, 0x06 },
	{ 0x800093, 0x00 },
	{ 0x800094, 0x00 },
	{ 0x800095, 0x00 },
	{ 0x800096, 0x00 },
	{ 0x800099, 0x01 },
	{ 0x80009b, 0x3c },
	{ 0x80009c, 0x28 },
	{ 0x80009f, 0xe1 },
	{ 0x8000a0, 0xcf },
	{ 0x8000a3, 0x01 },
	{ 0x8000a4, 0x5a },
	{ 0x8000a5, 0x01 },
	{ 0x8000a6, 0x01 },
	{ 0x8000a9, 0x00 },
	{ 0x8000aa, 0x01 },
	{ 0x8000b0, 0x01 },
	{ 0x8000b3, 0x02 },
	{ 0x8000b4, 0x32 },
	{ 0x8000b6, 0x14 },
	{ 0x8000c0, 0x11 },
	{ 0x8000c1, 0x00 },
	{ 0x8000c2, 0x05 },
	{ 0x8000c4, 0x00 },
	{ 0x8000c6, 0x19 },
	{ 0x8000c7, 0x00 },
	{ 0x8000cc, 0x2e },
	{ 0x8000cd, 0x51 },
	{ 0x8000ce, 0x33 },
	{ 0x8000f3, 0x05 },
	{ 0x8000f4, 0x8c },
	{ 0x8000f5, 0x8c },
	{ 0x8000f8, 0x03 },
	{ 0x8000f9, 0x06 },
	{ 0x8000fa, 0x06 },
	{ 0x8000fc, 0x02 },
	{ 0x8000fd, 0x02 },
	{ 0x8000fe, 0x02 },
	{ 0x8000ff, 0x09 },
	{ 0x800100, 0x50 },
	{ 0x800101, 0x7b },
	{ 0x800102, 0x77 },
	{ 0x800103, 0x00 },
	{ 0x800104, 0x02 },
	{ 0x800105, 0xc8 },
	{ 0x800106, 0x05 },
	{ 0x800107, 0x7b },
	{ 0x800109, 0x02 },
	{ 0x800115, 0x0a },
	{ 0x800116, 0x03 },
	{ 0x800117, 0x02 },
	{ 0x800118, 0x80 },
	{ 0x80011a, 0xc8 },
	{ 0x80011b, 0x7b },
	{ 0x80011c, 0x8a },
	{ 0x80011d, 0xa0 },
	{ 0x800122, 0x02 },
	{ 0x800123, 0x18 },
	{ 0x800124, 0xc3 },
	{ 0x800127, 0x00 },
	{ 0x800128, 0x07 },
	{ 0x80012a, 0x53 },
	{ 0x80012b, 0x51 },
	{ 0x80012c, 0x4e },
	{ 0x80012d, 0x43 },
	{ 0x800137, 0x01 },
	{ 0x800138, 0x00 },
	{ 0x800139, 0x07 },
	{ 0x80013a, 0x00 },
	{ 0x80013b, 0x06 },
	{ 0x80013d, 0x00 },
	{ 0x80013e, 0x01 },
	{ 0x80013f, 0x5b },
	{ 0x800140, 0xc8 },
	{ 0x800141, 0x59 },
	{ 0x80f000, 0x0f },
	{ 0x80f016, 0x10 },
	{ 0x80f017, 0x04 },
	{ 0x80f018, 0x05 },
	{ 0x80f019, 0x04 },
	{ 0x80f01a, 0x05 },
	{ 0x80f01f, 0x8c },
	{ 0x80f020, 0x00 },
	{ 0x80f021, 0x03 },
	{ 0x80f022, 0x0a },
	{ 0x80f023, 0x0a },
	{ 0x80f029, 0x8c },
	{ 0x80f02a, 0x00 },
	{ 0x80f02b, 0x00 },
	{ 0x80f02c, 0x01 },
	{ 0x80f064, 0x03 },
	{ 0x80f065, 0xf9 },
	{ 0x80f066, 0x03 },
	{ 0x80f067, 0x01 },
	{ 0x80f06f, 0xe0 },
	{ 0x80f070, 0x03 },
	{ 0x80f072, 0x0f },
	{ 0x80f073, 0x03 },
	{ 0x80f077, 0x01 },
	{ 0x80f078, 0x00 },
	{ 0x80f085, 0x00 },
	{ 0x80f086, 0x02 },
	{ 0x80f087, 0x00 },
	{ 0x80f09b, 0x3f },
	{ 0x80f09c, 0x00 },
	{ 0x80f09d, 0x20 },
	{ 0x80f09e, 0x00 },
	{ 0x80f09f, 0x0c },
	{ 0x80f0a0, 0x00 },
	{ 0x80f130, 0x04 },
	{ 0x80f132, 0x04 },
	{ 0x80f144, 0x1a },
	{ 0x80f146, 0x00 },
	{ 0x80f14a, 0x01 },
	{ 0x80f14c, 0x00 },
	{ 0x80f14d, 0x00 },
	{ 0x80f14f, 0x04 },
	{ 0x80f158, 0x7f },
	{ 0x80f15a, 0x00 },
	{ 0x80f15b, 0x08 },
	{ 0x80f15d, 0x03 },
	{ 0x80f15e, 0x05 },
	{ 0x80f163, 0x05 },
	{ 0x80f166, 0x01 },
	{ 0x80f167, 0x40 },
	{ 0x80f168, 0x0f },
	{ 0x80f17a, 0x00 },
	{ 0x80f17b, 0x00 },
	{ 0x80f183, 0x01 },
	{ 0x80f19d, 0x40 },
	{ 0x80f1bc, 0x36 },
	{ 0x80f1bd, 0x00 },
	{ 0x80f1cb, 0xa0 },
	{ 0x80f1cc, 0x01 },
	{ 0x80f204, 0x10 },
	{ 0x80f214, 0x00 },
	{ 0x80f24c, 0x88 },
	{ 0x80f24d, 0x95 },
	{ 0x80f24e, 0x9a },
	{ 0x80f24f, 0x90 },
	{ 0x80f25a, 0x07 },
	{ 0x80f25b, 0xe8 },
	{ 0x80f25c, 0x03 },
	{ 0x80f25d, 0xb0 },
	{ 0x80f25e, 0x04 },
	{ 0x80f270, 0x01 },
	{ 0x80f271, 0x02 },
	{ 0x80f272, 0x01 },
	{ 0x80f273, 0x02 },
	{ 0x80f40e, 0x0a },
	{ 0x80f40f, 0x40 },
	{ 0x80f410, 0x08 },
	{ 0x80f55f, 0x0a },
	{ 0x80f561, 0x15 },
	{ 0x80f562, 0x20 },
	{ 0x80f5df, 0xfb },
	{ 0x80f5e0, 0x00 },
	{ 0x80f5e3, 0x09 },
	{ 0x80f5e4, 0x01 },
	{ 0x80f5e5, 0x01 },
	{ 0x80f5f8, 0x01 },
	{ 0x80f5fd, 0x01 },
	{ 0x80f600, 0x05 },
	{ 0x80f601, 0x08 },
	{ 0x80f602, 0x0b },
	{ 0x80f603, 0x0e },
	{ 0x80f604, 0x11 },
	{ 0x80f605, 0x14 },
	{ 0x80f606, 0x17 },
	{ 0x80f607, 0x1f },
	{ 0x80f60e, 0x00 },
	{ 0x80f60f, 0x04 },
	{ 0x80f610, 0x32 },
	{ 0x80f611, 0x10 },
	{ 0x80f707, 0xfc },
	{ 0x80f708, 0x00 },
	{ 0x80f709, 0x37 },
	{ 0x80f70a, 0x00 },
	{ 0x80f78b, 0x01 },
	{ 0x80f80f, 0x40 },
	{ 0x80f810, 0x54 },
	{ 0x80f811, 0x5a },
	{ 0x80f905, 0x01 },
	{ 0x80fb06, 0x03 },
	{ 0x80fd8b, 0x00 },
};

/*
 * ITE Tech IT9133 AX Omega LNA config 1 tuner init
 * AF9033_TUNER_IT9135_51   = 0x51
 */
static const struct reg_val tuner_init_it9135_51[] = {
	{ 0x800043, 0x00 },
	{ 0x800046, 0x51 },
	{ 0x800051, 0x01 },
	{ 0x80005f, 0x00 },
	{ 0x800060, 0x00 },
	{ 0x800068, 0x0a },
	{ 0x800070, 0x0a },
	{ 0x800071, 0x06 },
	{ 0x800072, 0x02 },
	{ 0x800075, 0x8c },
	{ 0x800076, 0x8c },
	{ 0x800077, 0x8c },
	{ 0x800078, 0xc8 },
	{ 0x800079, 0x01 },
	{ 0x80007e, 0x04 },
	{ 0x80007f, 0x00 },
	{ 0x800081, 0x0a },
	{ 0x800082, 0x12 },
	{ 0x800083, 0x02 },
	{ 0x800084, 0x0a },
	{ 0x800085, 0x03 },
	{ 0x800086, 0xc0 },
	{ 0x800087, 0x96 },
	{ 0x800088, 0xcf },
	{ 0x800089, 0xc3 },
	{ 0x80008a, 0x01 },
	{ 0x80008e, 0x01 },
	{ 0x800092, 0x06 },
	{ 0x800093, 0x00 },
	{ 0x800094, 0x00 },
	{ 0x800095, 0x00 },
	{ 0x800096, 0x00 },
	{ 0x800099, 0x01 },
	{ 0x80009b, 0x3c },
	{ 0x80009c, 0x28 },
	{ 0x80009f, 0xe1 },
	{ 0x8000a0, 0xcf },
	{ 0x8000a3, 0x01 },
	{ 0x8000a4, 0x5a },
	{ 0x8000a5, 0x01 },
	{ 0x8000a6, 0x01 },
	{ 0x8000a9, 0x00 },
	{ 0x8000aa, 0x01 },
	{ 0x8000b0, 0x01 },
	{ 0x8000b3, 0x02 },
	{ 0x8000b4, 0x3c },
	{ 0x8000b6, 0x14 },
	{ 0x8000c0, 0x11 },
	{ 0x8000c1, 0x00 },
	{ 0x8000c2, 0x05 },
	{ 0x8000c4, 0x00 },
	{ 0x8000c6, 0x19 },
	{ 0x8000c7, 0x00 },
	{ 0x8000cc, 0x2e },
	{ 0x8000cd, 0x51 },
	{ 0x8000ce, 0x33 },
	{ 0x8000f3, 0x05 },
	{ 0x8000f4, 0x8c },
	{ 0x8000f5, 0x8c },
	{ 0x8000f8, 0x03 },
	{ 0x8000f9, 0x06 },
	{ 0x8000fa, 0x06 },
	{ 0x8000fc, 0x03 },
	{ 0x8000fd, 0x02 },
	{ 0x8000fe, 0x02 },
	{ 0x8000ff, 0x09 },
	{ 0x800100, 0x50 },
	{ 0x800101, 0x7a },
	{ 0x800102, 0x77 },
	{ 0x800103, 0x01 },
	{ 0x800104, 0x02 },
	{ 0x800105, 0xb0 },
	{ 0x800106, 0x02 },
	{ 0x800107, 0x7a },
	{ 0x800109, 0x02 },
	{ 0x800115, 0x0a },
	{ 0x800116, 0x03 },
	{ 0x800117, 0x02 },
	{ 0x800118, 0x80 },
	{ 0x80011a, 0xc0 },
	{ 0x80011b, 0x7a },
	{ 0x80011c, 0xac },
	{ 0x80011d, 0x8c },
	{ 0x800122, 0x02 },
	{ 0x800123, 0x70 },
	{ 0x800124, 0xa4 },
	{ 0x800127, 0x00 },
	{ 0x800128, 0x07 },
	{ 0x80012a, 0x53 },
	{ 0x80012b, 0x51 },
	{ 0x80012c, 0x4e },
	{ 0x80012d, 0x43 },
	{ 0x800137, 0x01 },
	{ 0x800138, 0x00 },
	{ 0x800139, 0x07 },
	{ 0x80013a, 0x00 },
	{ 0x80013b, 0x06 },
	{ 0x80013d, 0x00 },
	{ 0x80013e, 0x01 },
	{ 0x80013f, 0x5b },
	{ 0x800140, 0xc0 },
	{ 0x800141, 0x59 },
	{ 0x80f000, 0x0f },
	{ 0x80f016, 0x10 },
	{ 0x80f017, 0x04 },
	{ 0x80f018, 0x05 },
	{ 0x80f019, 0x04 },
	{ 0x80f01a, 0x05 },
	{ 0x80f01f, 0x8c },
	{ 0x80f020, 0x00 },
	{ 0x80f021, 0x03 },
	{ 0x80f022, 0x0a },
	{ 0x80f023, 0x0a },
	{ 0x80f029, 0x8c },
	{ 0x80f02a, 0x00 },
	{ 0x80f02b, 0x00 },
	{ 0x80f02c, 0x01 },
	{ 0x80f064, 0x03 },
	{ 0x80f065, 0xf9 },
	{ 0x80f066, 0x03 },
	{ 0x80f067, 0x01 },
	{ 0x80f06f, 0xe0 },
	{ 0x80f070, 0x03 },
	{ 0x80f072, 0x0f },
	{ 0x80f073, 0x03 },
	{ 0x80f077, 0x01 },
	{ 0x80f078, 0x00 },
	{ 0x80f085, 0xc0 },
	{ 0x80f086, 0x01 },
	{ 0x80f087, 0x00 },
	{ 0x80f09b, 0x3f },
	{ 0x80f09c, 0x00 },
	{ 0x80f09d, 0x20 },
	{ 0x80f09e, 0x00 },
	{ 0x80f09f, 0x0c },
	{ 0x80f0a0, 0x00 },
	{ 0x80f130, 0x04 },
	{ 0x80f132, 0x04 },
	{ 0x80f144, 0x1a },
	{ 0x80f146, 0x00 },
	{ 0x80f14a, 0x01 },
	{ 0x80f14c, 0x00 },
	{ 0x80f14d, 0x00 },
	{ 0x80f14f, 0x04 },
	{ 0x80f158, 0x7f },
	{ 0x80f15a, 0x00 },
	{ 0x80f15b, 0x08 },
	{ 0x80f15d, 0x03 },
	{ 0x80f15e, 0x05 },
	{ 0x80f163, 0x05 },
	{ 0x80f166, 0x01 },
	{ 0x80f167, 0x40 },
	{ 0x80f168, 0x0f },
	{ 0x80f17a, 0x00 },
	{ 0x80f17b, 0x00 },
	{ 0x80f183, 0x01 },
	{ 0x80f19d, 0x40 },
	{ 0x80f1bc, 0x36 },
	{ 0x80f1bd, 0x00 },
	{ 0x80f1cb, 0xa0 },
	{ 0x80f1cc, 0x01 },
	{ 0x80f204, 0x10 },
	{ 0x80f214, 0x00 },
	{ 0x80f24c, 0x88 },
	{ 0x80f24d, 0x95 },
	{ 0x80f24e, 0x9a },
	{ 0x80f24f, 0x90 },
	{ 0x80f25a, 0x07 },
	{ 0x80f25b, 0xe8 },
	{ 0x80f25c, 0x03 },
	{ 0x80f25d, 0xb0 },
	{ 0x80f25e, 0x04 },
	{ 0x80f270, 0x01 },
	{ 0x80f271, 0x02 },
	{ 0x80f272, 0x01 },
	{ 0x80f273, 0x02 },
	{ 0x80f40e, 0x0a },
	{ 0x80f40f, 0x40 },
	{ 0x80f410, 0x08 },
	{ 0x80f55f, 0x0a },
	{ 0x80f561, 0x15 },
	{ 0x80f562, 0x20 },
	{ 0x80f5df, 0xfb },
	{ 0x80f5e0, 0x00 },
	{ 0x80f5e3, 0x09 },
	{ 0x80f5e4, 0x01 },
	{ 0x80f5e5, 0x01 },
	{ 0x80f5f8, 0x01 },
	{ 0x80f5fd, 0x01 },
	{ 0x80f600, 0x05 },
	{ 0x80f601, 0x08 },
	{ 0x80f602, 0x0b },
	{ 0x80f603, 0x0e },
	{ 0x80f604, 0x11 },
	{ 0x80f605, 0x14 },
	{ 0x80f606, 0x17 },
	{ 0x80f607, 0x1f },
	{ 0x80f60e, 0x00 },
	{ 0x80f60f, 0x04 },
	{ 0x80f610, 0x32 },
	{ 0x80f611, 0x10 },
	{ 0x80f707, 0xfc },
	{ 0x80f708, 0x00 },
	{ 0x80f709, 0x37 },
	{ 0x80f70a, 0x00 },
	{ 0x80f78b, 0x01 },
	{ 0x80f80f, 0x40 },
	{ 0x80f810, 0x54 },
	{ 0x80f811, 0x5a },
	{ 0x80f905, 0x01 },
	{ 0x80fb06, 0x03 },
	{ 0x80fd8b, 0x00 },
};

/*
 * ITE Tech IT9133 AX Omega LNA config 2 tuner init
 * AF9033_TUNER_IT9135_52   = 0x52
 */
static const struct reg_val tuner_init_it9135_52[] = {
	{ 0x800043, 0x00 },
	{ 0x800046, 0x52 },
	{ 0x800051, 0x01 },
	{ 0x80005f, 0x00 },
	{ 0x800060, 0x00 },
	{ 0x800068, 0x10 },
	{ 0x800070, 0x0a },
	{ 0x800071, 0x05 },
	{ 0x800072, 0x02 },
	{ 0x800075, 0x8c },
	{ 0x800076, 0x8c },
	{ 0x800077, 0x8c },
	{ 0x800078, 0xa0 },
	{ 0x800079, 0x01 },
	{ 0x80007e, 0x04 },
	{ 0x80007f, 0x00 },
	{ 0x800081, 0x0a },
	{ 0x800082, 0x17 },
	{ 0x800083, 0x03 },
	{ 0x800084, 0x0a },
	{ 0x800085, 0x03 },
	{ 0x800086, 0xb3 },
	{ 0x800087, 0x97 },
	{ 0x800088, 0xc0 },
	{ 0x800089, 0x9e },
	{ 0x80008a, 0x01 },
	{ 0x80008e, 0x01 },
	{ 0x800092, 0x06 },
	{ 0x800093, 0x00 },
	{ 0x800094, 0x00 },
	{ 0x800095, 0x00 },
	{ 0x800096, 0x00 },
	{ 0x800099, 0x01 },
	{ 0x80009b, 0x3c },
	{ 0x80009c, 0x28 },
	{ 0x80009f, 0xe1 },
	{ 0x8000a0, 0xcf },
	{ 0x8000a3, 0x01 },
	{ 0x8000a4, 0x5c },
	{ 0x8000a5, 0x01 },
	{ 0x8000a6, 0x01 },
	{ 0x8000a9, 0x00 },
	{ 0x8000aa, 0x01 },
	{ 0x8000b0, 0x01 },
	{ 0x8000b3, 0x02 },
	{ 0x8000b4, 0x3c },
	{ 0x8000b6, 0x14 },
	{ 0x8000c0, 0x11 },
	{ 0x8000c1, 0x00 },
	{ 0x8000c2, 0x05 },
	{ 0x8000c4, 0x00 },
	{ 0x8000c6, 0x19 },
	{ 0x8000c7, 0x00 },
	{ 0x8000cc, 0x2e },
	{ 0x8000cd, 0x51 },
	{ 0x8000ce, 0x33 },
	{ 0x8000f3, 0x05 },
	{ 0x8000f4, 0x91 },
	{ 0x8000f5, 0x8c },
	{ 0x8000f8, 0x03 },
	{ 0x8000f9, 0x06 },
	{ 0x8000fa, 0x06 },
	{ 0x8000fc, 0x03 },
	{ 0x8000fd, 0x02 },
	{ 0x8000fe, 0x02 },
	{ 0x8000ff, 0x09 },
	{ 0x800100, 0x50 },
	{ 0x800101, 0x74 },
	{ 0x800102, 0x77 },
	{ 0x800103, 0x02 },
	{ 0x800104, 0x02 },
	{ 0x800105, 0xa4 },
	{ 0x800106, 0x02 },
	{ 0x800107, 0x6e },
	{ 0x800109, 0x02 },
	{ 0x800115, 0x0a },
	{ 0x800116, 0x03 },
	{ 0x800117, 0x02 },
	{ 0x800118, 0x80 },
	{ 0x80011a, 0xcd },
	{ 0x80011b, 0x62 },
	{ 0x80011c, 0xa4 },
	{ 0x80011d, 0x8c },
	{ 0x800122, 0x03 },
	{ 0x800123, 0x18 },
	{ 0x800124, 0x9e },
	{ 0x800127, 0x00 },
	{ 0x800128, 0x07 },
	{ 0x80012a, 0x53 },
	{ 0x80012b, 0x51 },
	{ 0x80012c, 0x4e },
	{ 0x80012d, 0x43 },
	{ 0x800137, 0x00 },
	{ 0x800138, 0x00 },
	{ 0x800139, 0x07 },
	{ 0x80013a, 0x00 },
	{ 0x80013b, 0x06 },
	{ 0x80013d, 0x00 },
	{ 0x80013e, 0x01 },
	{ 0x80013f, 0x5b },
	{ 0x800140, 0xb6 },
	{ 0x800141, 0x59 },
	{ 0x80f000, 0x0f },
	{ 0x80f016, 0x10 },
	{ 0x80f017, 0x04 },
	{ 0x80f018, 0x05 },
	{ 0x80f019, 0x04 },
	{ 0x80f01a, 0x05 },
	{ 0x80f01f, 0x8c },
	{ 0x80f020, 0x00 },
	{ 0x80f021, 0x03 },
	{ 0x80f022, 0x0a },
	{ 0x80f023, 0x0a },
	{ 0x80f029, 0x8c },
	{ 0x80f02a, 0x00 },
	{ 0x80f02b, 0x00 },
	{ 0x80f02c, 0x01 },
	{ 0x80f064, 0x03 },
	{ 0x80f065, 0xf9 },
	{ 0x80f066, 0x03 },
	{ 0x80f067, 0x01 },
	{ 0x80f06f, 0xe0 },
	{ 0x80f070, 0x03 },
	{ 0x80f072, 0x0f },
	{ 0x80f073, 0x03 },
	{ 0x80f077, 0x01 },
	{ 0x80f078, 0x00 },
	{ 0x80f085, 0xc0 },
	{ 0x80f086, 0x01 },
	{ 0x80f087, 0x00 },
	{ 0x80f09b, 0x3f },
	{ 0x80f09c, 0x00 },
	{ 0x80f09d, 0x20 },
	{ 0x80f09e, 0x00 },
	{ 0x80f09f, 0x0c },
	{ 0x80f0a0, 0x00 },
	{ 0x80f130, 0x04 },
	{ 0x80f132, 0x04 },
	{ 0x80f144, 0x1a },
	{ 0x80f146, 0x00 },
	{ 0x80f14a, 0x01 },
	{ 0x80f14c, 0x00 },
	{ 0x80f14d, 0x00 },
	{ 0x80f14f, 0x04 },
	{ 0x80f158, 0x7f },
	{ 0x80f15a, 0x00 },
	{ 0x80f15b, 0x08 },
	{ 0x80f15d, 0x03 },
	{ 0x80f15e, 0x05 },
	{ 0x80f163, 0x05 },
	{ 0x80f166, 0x01 },
	{ 0x80f167, 0x40 },
	{ 0x80f168, 0x0f },
	{ 0x80f17a, 0x00 },
	{ 0x80f17b, 0x00 },
	{ 0x80f183, 0x01 },
	{ 0x80f19d, 0x40 },
	{ 0x80f1bc, 0x36 },
	{ 0x80f1bd, 0x00 },
	{ 0x80f1cb, 0xa0 },
	{ 0x80f1cc, 0x01 },
	{ 0x80f204, 0x10 },
	{ 0x80f214, 0x00 },
	{ 0x80f24c, 0x88 },
	{ 0x80f24d, 0x95 },
	{ 0x80f24e, 0x9a },
	{ 0x80f24f, 0x90 },
	{ 0x80f25a, 0x07 },
	{ 0x80f25b, 0xe8 },
	{ 0x80f25c, 0x03 },
	{ 0x80f25d, 0xb0 },
	{ 0x80f25e, 0x04 },
	{ 0x80f270, 0x01 },
	{ 0x80f271, 0x02 },
	{ 0x80f272, 0x01 },
	{ 0x80f273, 0x02 },
	{ 0x80f40e, 0x0a },
	{ 0x80f40f, 0x40 },
	{ 0x80f410, 0x08 },
	{ 0x80f55f, 0x0a },
	{ 0x80f561, 0x15 },
	{ 0x80f562, 0x20 },
	{ 0x80f5df, 0xfb },
	{ 0x80f5e0, 0x00 },
	{ 0x80f5e3, 0x09 },
	{ 0x80f5e4, 0x01 },
	{ 0x80f5e5, 0x01 },
	{ 0x80f5f8, 0x01 },
	{ 0x80f5fd, 0x01 },
	{ 0x80f600, 0x05 },
	{ 0x80f601, 0x08 },
	{ 0x80f602, 0x0b },
	{ 0x80f603, 0x0e },
	{ 0x80f604, 0x11 },
	{ 0x80f605, 0x14 },
	{ 0x80f606, 0x17 },
	{ 0x80f607, 0x1f },
	{ 0x80f60e, 0x00 },
	{ 0x80f60f, 0x04 },
	{ 0x80f610, 0x32 },
	{ 0x80f611, 0x10 },
	{ 0x80f707, 0xfc },
	{ 0x80f708, 0x00 },
	{ 0x80f709, 0x37 },
	{ 0x80f70a, 0x00 },
	{ 0x80f78b, 0x01 },
	{ 0x80f80f, 0x40 },
	{ 0x80f810, 0x54 },
	{ 0x80f811, 0x5a },
	{ 0x80f905, 0x01 },
	{ 0x80fb06, 0x03 },
	{ 0x80fd8b, 0x00 },
};

/*
 * ITE Tech IT9133 BX demod init
 */
static const struct reg_val ofsm_init_it9135_v2[] = {
	{ 0x800051, 0x01 },
	{ 0x800070, 0x0a },
	{ 0x80007e, 0x04 },
	{ 0x800081, 0x0a },
	{ 0x80008a, 0x01 },
	{ 0x80008e, 0x01 },
	{ 0x800092, 0x06 },
	{ 0x800099, 0x01 },
	{ 0x80009f, 0xe1 },
	{ 0x8000a0, 0xcf },
	{ 0x8000a3, 0x01 },
	{ 0x8000a5, 0x01 },
	{ 0x8000a6, 0x01 },
	{ 0x8000a9, 0x00 },
	{ 0x8000aa, 0x01 },
	{ 0x8000b0, 0x01 },
	{ 0x8000c2, 0x05 },
	{ 0x8000c6, 0x19 },
	{ 0x80f000, 0x0f },
	{ 0x80f02b, 0x00 },
	{ 0x80f064, 0x03 },
	{ 0x80f065, 0xf9 },
	{ 0x80f066, 0x03 },
	{ 0x80f067, 0x01 },
	{ 0x80f06f, 0xe0 },
	{ 0x80f070, 0x03 },
	{ 0x80f072, 0x0f },
	{ 0x80f073, 0x03 },
	{ 0x80f078, 0x00 },
	{ 0x80f087, 0x00 },
	{ 0x80f09b, 0x3f },
	{ 0x80f09c, 0x00 },
	{ 0x80f09d, 0x20 },
	{ 0x80f09e, 0x00 },
	{ 0x80f09f, 0x0c },
	{ 0x80f0a0, 0x00 },
	{ 0x80f130, 0x04 },
	{ 0x80f132, 0x04 },
	{ 0x80f144, 0x1a },
	{ 0x80f146, 0x00 },
	{ 0x80f14a, 0x01 },
	{ 0x80f14c, 0x00 },
	{ 0x80f14d, 0x00 },
	{ 0x80f14f, 0x04 },
	{ 0x80f158, 0x7f },
	{ 0x80f15a, 0x00 },
	{ 0x80f15b, 0x08 },
	{ 0x80f15d, 0x03 },
	{ 0x80f15e, 0x05 },
	{ 0x80f163, 0x05 },
	{ 0x80f166, 0x01 },
	{ 0x80f167, 0x40 },
	{ 0x80f168, 0x0f },
	{ 0x80f17a, 0x00 },
	{ 0x80f17b, 0x00 },
	{ 0x80f183, 0x01 },
	{ 0x80f19d, 0x40 },
	{ 0x80f1bc, 0x36 },
	{ 0x80f1bd, 0x00 },
	{ 0x80f1cb, 0xa0 },
	{ 0x80f1cc, 0x01 },
	{ 0x80f204, 0x10 },
	{ 0x80f214, 0x00 },
	{ 0x80f40e, 0x0a },
	{ 0x80f40f, 0x40 },
	{ 0x80f410, 0x08 },
	{ 0x80f55f, 0x0a },
	{ 0x80f561, 0x15 },
	{ 0x80f562, 0x20 },
	{ 0x80f5e3, 0x09 },
	{ 0x80f5e4, 0x01 },
	{ 0x80f5e5, 0x01 },
	{ 0x80f600, 0x05 },
	{ 0x80f601, 0x08 },
	{ 0x80f602, 0x0b },
	{ 0x80f603, 0x0e },
	{ 0x80f604, 0x11 },
	{ 0x80f605, 0x14 },
	{ 0x80f606, 0x17 },
	{ 0x80f607, 0x1f },
	{ 0x80f60e, 0x00 },
	{ 0x80f60f, 0x04 },
	{ 0x80f610, 0x32 },
	{ 0x80f611, 0x10 },
	{ 0x80f707, 0xfc },
	{ 0x80f708, 0x00 },
	{ 0x80f709, 0x37 },
	{ 0x80f70a, 0x00 },
	{ 0x80f78b, 0x01 },
	{ 0x80f80f, 0x40 },
	{ 0x80f810, 0x54 },
	{ 0x80f811, 0x5a },
	{ 0x80f905, 0x01 },
	{ 0x80fb06, 0x03 },
	{ 0x80fd8b, 0x00 },
};

/*
 * ITE Tech IT9133 BX Omega tuner init
 * AF9033_TUNER_IT9135_60   = 0x60
 */
static const struct reg_val tuner_init_it9135_60[] = {
	{ 0x800043, 0x00 },
	{ 0x800046, 0x60 },
	{ 0x800051, 0x01 },
	{ 0x80005f, 0x00 },
	{ 0x800060, 0x00 },
	{ 0x800068, 0x0a },
	{ 0x80006a, 0x03 },
	{ 0x800070, 0x0a },
	{ 0x800071, 0x0a },
	{ 0x800072, 0x02 },
	{ 0x800075, 0x8c },
	{ 0x800076, 0x8c },
	{ 0x800077, 0x8c },
	{ 0x800078, 0x8c },
	{ 0x800079, 0x01 },
	{ 0x80007e, 0x04 },
	{ 0x800081, 0x0a },
	{ 0x800082, 0x18 },
	{ 0x800084, 0x0a },
	{ 0x800085, 0x33 },
	{ 0x800086, 0xbe },
	{ 0x800087, 0xa0 },
	{ 0x800088, 0xc6 },
	{ 0x800089, 0xb6 },
	{ 0x80008a, 0x01 },
	{ 0x80008e, 0x01 },
	{ 0x800092, 0x06 },
	{ 0x800093, 0x00 },
	{ 0x800094, 0x00 },
	{ 0x800095, 0x00 },
	{ 0x800096, 0x00 },
	{ 0x800099, 0x01 },
	{ 0x80009b, 0x3c },
	{ 0x80009c, 0x28 },
	{ 0x80009f, 0xe1 },
	{ 0x8000a0, 0xcf },
	{ 0x8000a3, 0x01 },
	{ 0x8000a4, 0x5a },
	{ 0x8000a5, 0x01 },
	{ 0x8000a6, 0x01 },
	{ 0x8000a9, 0x00 },
	{ 0x8000aa, 0x01 },
	{ 0x8000b0, 0x01 },
	{ 0x8000b3, 0x02 },
	{ 0x8000b4, 0x3a },
	{ 0x8000b6, 0x14 },
	{ 0x8000c0, 0x11 },
	{ 0x8000c1, 0x00 },
	{ 0x8000c2, 0x05 },
	{ 0x8000c3, 0x01 },
	{ 0x8000c4, 0x00 },
	{ 0x8000c6, 0x19 },
	{ 0x8000c7, 0x00 },
	{ 0x8000cb, 0x32 },
	{ 0x8000cc, 0x2c },
	{ 0x8000cd, 0x4f },
	{ 0x8000ce, 0x30 },
	{ 0x8000f3, 0x05 },
	{ 0x8000f4, 0xa0 },
	{ 0x8000f5, 0x8c },
	{ 0x8000f8, 0x03 },
	{ 0x8000f9, 0x06 },
	{ 0x8000fa, 0x06 },
	{ 0x8000fc, 0x03 },
	{ 0x8000fd, 0x03 },
	{ 0x8000fe, 0x02 },
	{ 0x8000ff, 0x0a },
	{ 0x800100, 0x50 },
	{ 0x800101, 0x7b },
	{ 0x800102, 0x8c },
	{ 0x800103, 0x00 },
	{ 0x800104, 0x02 },
	{ 0x800105, 0xbe },
	{ 0x800106, 0x00 },
	{ 0x800115, 0x0a },
	{ 0x800116, 0x03 },
	{ 0x80011a, 0xbe },
	{ 0x800124, 0xae },
	{ 0x800127, 0x00 },
	{ 0x80012a, 0x56 },
	{ 0x80012b, 0x50 },
	{ 0x80012c, 0x47 },
	{ 0x80012d, 0x42 },
	{ 0x800137, 0x00 },
	{ 0x80013b, 0x08 },
	{ 0x80013f, 0x5b },
	{ 0x800141, 0x59 },
	{ 0x800142, 0xf9 },
	{ 0x800143, 0x19 },
	{ 0x800144, 0x00 },
	{ 0x800145, 0x8c },
	{ 0x800146, 0x8c },
	{ 0x800147, 0x8c },
	{ 0x800148, 0x6e },
	{ 0x800149, 0x8c },
	{ 0x80014a, 0x50 },
	{ 0x80014b, 0x8c },
	{ 0x80014d, 0xac },
	{ 0x80014e, 0xc6 },
	{ 0x800151, 0x1e },
	{ 0x800153, 0xbc },
	{ 0x800178, 0x09 },
	{ 0x800181, 0x94 },
	{ 0x800182, 0x6e },
	{ 0x800185, 0x24 },
	{ 0x800189, 0xbe },
	{ 0x80018c, 0x03 },
	{ 0x80018d, 0x5f },
	{ 0x80018f, 0xa0 },
	{ 0x800190, 0x5a },
	{ 0x800191, 0x00 },
	{ 0x80ed02, 0x40 },
	{ 0x80ee42, 0x40 },
	{ 0x80ee82, 0x40 },
	{ 0x80f000, 0x0f },
	{ 0x80f01f, 0x8c },
	{ 0x80f020, 0x00 },
	{ 0x80f029, 0x8c },
	{ 0x80f02a, 0x00 },
	{ 0x80f02b, 0x00 },
	{ 0x80f064, 0x03 },
	{ 0x80f065, 0xf9 },
	{ 0x80f066, 0x03 },
	{ 0x80f067, 0x01 },
	{ 0x80f06f, 0xe0 },
	{ 0x80f070, 0x03 },
	{ 0x80f072, 0x0f },
	{ 0x80f073, 0x03 },
	{ 0x80f077, 0x01 },
	{ 0x80f078, 0x00 },
	{ 0x80f087, 0x00 },
	{ 0x80f09b, 0x3f },
	{ 0x80f09c, 0x00 },
	{ 0x80f09d, 0x20 },
	{ 0x80f09e, 0x00 },
	{ 0x80f09f, 0x0c },
	{ 0x80f0a0, 0x00 },
	{ 0x80f130, 0x04 },
	{ 0x80f132, 0x04 },
	{ 0x80f144, 0x1a },
	{ 0x80f146, 0x00 },
	{ 0x80f14a, 0x01 },
	{ 0x80f14c, 0x00 },
	{ 0x80f14d, 0x00 },
	{ 0x80f14f, 0x04 },
	{ 0x80f158, 0x7f },
	{ 0x80f15a, 0x00 },
	{ 0x80f15b, 0x08 },
	{ 0x80f15d, 0x03 },
	{ 0x80f15e, 0x05 },
	{ 0x80f163, 0x05 },
	{ 0x80f166, 0x01 },
	{ 0x80f167, 0x40 },
	{ 0x80f168, 0x0f },
	{ 0x80f17a, 0x00 },
	{ 0x80f17b, 0x00 },
	{ 0x80f183, 0x01 },
	{ 0x80f19d, 0x40 },
	{ 0x80f1bc, 0x36 },
	{ 0x80f1bd, 0x00 },
	{ 0x80f1cb, 0xa0 },
	{ 0x80f1cc, 0x01 },
	{ 0x80f204, 0x10 },
	{ 0x80f214, 0x00 },
	{ 0x80f24c, 0x88 },
	{ 0x80f24d, 0x95 },
	{ 0x80f24e, 0x9a },
	{ 0x80f24f, 0x90 },
	{ 0x80f25a, 0x07 },
	{ 0x80f25b, 0xe8 },
	{ 0x80f25c, 0x03 },
	{ 0x80f25d, 0xb0 },
	{ 0x80f25e, 0x04 },
	{ 0x80f270, 0x01 },
	{ 0x80f271, 0x02 },
	{ 0x80f272, 0x01 },
	{ 0x80f273, 0x02 },
	{ 0x80f40e, 0x0a },
	{ 0x80f40f, 0x40 },
	{ 0x80f410, 0x08 },
	{ 0x80f55f, 0x0a },
	{ 0x80f561, 0x15 },
	{ 0x80f562, 0x20 },
	{ 0x80f5e3, 0x09 },
	{ 0x80f5e4, 0x01 },
	{ 0x80f5e5, 0x01 },
	{ 0x80f600, 0x05 },
	{ 0x80f601, 0x08 },
	{ 0x80f602, 0x0b },
	{ 0x80f603, 0x0e },
	{ 0x80f604, 0x11 },
	{ 0x80f605, 0x14 },
	{ 0x80f606, 0x17 },
	{ 0x80f607, 0x1f },
	{ 0x80f60e, 0x00 },
	{ 0x80f60f, 0x04 },
	{ 0x80f610, 0x32 },
	{ 0x80f611, 0x10 },
	{ 0x80f707, 0xfc },
	{ 0x80f708, 0x00 },
	{ 0x80f709, 0x37 },
	{ 0x80f70a, 0x00 },
	{ 0x80f78b, 0x01 },
	{ 0x80f80f, 0x40 },
	{ 0x80f810, 0x54 },
	{ 0x80f811, 0x5a },
	{ 0x80f905, 0x01 },
	{ 0x80fb06, 0x03 },
	{ 0x80fd8b, 0x00 },
};

/*
 * ITE Tech IT9133 BX Omega LNA config 1 tuner init
 * AF9033_TUNER_IT9135_61   = 0x61
 */
static const struct reg_val tuner_init_it9135_61[] = {
	{ 0x800043, 0x00 },
	{ 0x800046, 0x61 },
	{ 0x800051, 0x01 },
	{ 0x80005f, 0x00 },
	{ 0x800060, 0x00 },
	{ 0x800068, 0x06 },
	{ 0x80006a, 0x03 },
	{ 0x800070, 0x0a },
	{ 0x800071, 0x05 },
	{ 0x800072, 0x02 },
	{ 0x800075, 0x8c },
	{ 0x800076, 0x8c },
	{ 0x800077, 0x8c },
	{ 0x800078, 0x90 },
	{ 0x800079, 0x01 },
	{ 0x80007e, 0x04 },
	{ 0x800081, 0x0a },
	{ 0x800082, 0x12 },
	{ 0x800084, 0x0a },
	{ 0x800085, 0x33 },
	{ 0x800086, 0xbc },
	{ 0x800087, 0x9c },
	{ 0x800088, 0xcc },
	{ 0x800089, 0xa8 },
	{ 0x80008a, 0x01 },
	{ 0x80008e, 0x01 },
	{ 0x800092, 0x06 },
	{ 0x800093, 0x00 },
	{ 0x800094, 0x00 },
	{ 0x800095, 0x00 },
	{ 0x800096, 0x00 },
	{ 0x800099, 0x01 },
	{ 0x80009b, 0x3c },
	{ 0x80009c, 0x28 },
	{ 0x80009f, 0xe1 },
	{ 0x8000a0, 0xcf },
	{ 0x8000a3, 0x01 },
	{ 0x8000a4, 0x5c },
	{ 0x8000a5, 0x01 },
	{ 0x8000a6, 0x01 },
	{ 0x8000a9, 0x00 },
	{ 0x8000aa, 0x01 },
	{ 0x8000b0, 0x01 },
	{ 0x8000b3, 0x02 },
	{ 0x8000b4, 0x3a },
	{ 0x8000b6, 0x14 },
	{ 0x8000c0, 0x11 },
	{ 0x8000c1, 0x00 },
	{ 0x8000c2, 0x05 },
	{ 0x8000c3, 0x01 },
	{ 0x8000c4, 0x00 },
	{ 0x8000c6, 0x19 },
	{ 0x8000c7, 0x00 },
	{ 0x8000cb, 0x32 },
	{ 0x8000cc, 0x2c },
	{ 0x8000cd, 0x4f },
	{ 0x8000ce, 0x30 },
	{ 0x8000f3, 0x05 },
	{ 0x8000f4, 0xa0 },
	{ 0x8000f5, 0x8c },
	{ 0x8000f8, 0x03 },
	{ 0x8000f9, 0x06 },
	{ 0x8000fa, 0x06 },
	{ 0x8000fc, 0x03 },
	{ 0x8000fd, 0x03 },
	{ 0x8000fe, 0x02 },
	{ 0x8000ff, 0x08 },
	{ 0x800100, 0x50 },
	{ 0x800101, 0x7b },
	{ 0x800102, 0x8c },
	{ 0x800103, 0x01 },
	{ 0x800104, 0x02 },
	{ 0x800105, 0xc8 },
	{ 0x800106, 0x00 },
	{ 0x800115, 0x0a },
	{ 0x800116, 0x03 },
	{ 0x80011a, 0xc6 },
	{ 0x800124, 0xa8 },
	{ 0x800127, 0x00 },
	{ 0x80012a, 0x59 },
	{ 0x80012b, 0x50 },
	{ 0x80012c, 0x47 },
	{ 0x80012d, 0x42 },
	{ 0x800137, 0x00 },
	{ 0x80013b, 0x05 },
	{ 0x80013f, 0x5b },
	{ 0x800141, 0x59 },
	{ 0x800142, 0xf9 },
	{ 0x800143, 0x59 },
	{ 0x800144, 0x01 },
	{ 0x800145, 0x8c },
	{ 0x800146, 0x8c },
	{ 0x800147, 0x8c },
	{ 0x800148, 0x7b },
	{ 0x800149, 0x8c },
	{ 0x80014a, 0x50 },
	{ 0x80014b, 0x8c },
	{ 0x80014d, 0xa8 },
	{ 0x80014e, 0xc6 },
	{ 0x800151, 0x28 },
	{ 0x800153, 0xcc },
	{ 0x800178, 0x09 },
	{ 0x800181, 0x9c },
	{ 0x800182, 0x76 },
	{ 0x800185, 0x28 },
	{ 0x800189, 0xaa },
	{ 0x80018c, 0x03 },
	{ 0x80018d, 0x5f },
	{ 0x80018f, 0xfb },
	{ 0x800190, 0x5c },
	{ 0x800191, 0x00 },
	{ 0x80ed02, 0x40 },
	{ 0x80ee42, 0x40 },
	{ 0x80ee82, 0x40 },
	{ 0x80f000, 0x0f },
	{ 0x80f01f, 0x8c },
	{ 0x80f020, 0x00 },
	{ 0x80f029, 0x8c },
	{ 0x80f02a, 0x00 },
	{ 0x80f02b, 0x00 },
	{ 0x80f064, 0x03 },
	{ 0x80f065, 0xf9 },
	{ 0x80f066, 0x03 },
	{ 0x80f067, 0x01 },
	{ 0x80f06f, 0xe0 },
	{ 0x80f070, 0x03 },
	{ 0x80f072, 0x0f },
	{ 0x80f073, 0x03 },
	{ 0x80f077, 0x01 },
	{ 0x80f078, 0x00 },
	{ 0x80f087, 0x00 },
	{ 0x80f09b, 0x3f },
	{ 0x80f09c, 0x00 },
	{ 0x80f09d, 0x20 },
	{ 0x80f09e, 0x00 },
	{ 0x80f09f, 0x0c },
	{ 0x80f0a0, 0x00 },
	{ 0x80f130, 0x04 },
	{ 0x80f132, 0x04 },
	{ 0x80f144, 0x1a },
	{ 0x80f146, 0x00 },
	{ 0x80f14a, 0x01 },
	{ 0x80f14c, 0x00 },
	{ 0x80f14d, 0x00 },
	{ 0x80f14f, 0x04 },
	{ 0x80f158, 0x7f },
	{ 0x80f15a, 0x00 },
	{ 0x80f15b, 0x08 },
	{ 0x80f15d, 0x03 },
	{ 0x80f15e, 0x05 },
	{ 0x80f163, 0x05 },
	{ 0x80f166, 0x01 },
	{ 0x80f167, 0x40 },
	{ 0x80f168, 0x0f },
	{ 0x80f17a, 0x00 },
	{ 0x80f17b, 0x00 },
	{ 0x80f183, 0x01 },
	{ 0x80f19d, 0x40 },
	{ 0x80f1bc, 0x36 },
	{ 0x80f1bd, 0x00 },
	{ 0x80f1cb, 0xa0 },
	{ 0x80f1cc, 0x01 },
	{ 0x80f204, 0x10 },
	{ 0x80f214, 0x00 },
	{ 0x80f24c, 0x88 },
	{ 0x80f24d, 0x95 },
	{ 0x80f24e, 0x9a },
	{ 0x80f24f, 0x90 },
	{ 0x80f25a, 0x07 },
	{ 0x80f25b, 0xe8 },
	{ 0x80f25c, 0x03 },
	{ 0x80f25d, 0xb0 },
	{ 0x80f25e, 0x04 },
	{ 0x80f270, 0x01 },
	{ 0x80f271, 0x02 },
	{ 0x80f272, 0x01 },
	{ 0x80f273, 0x02 },
	{ 0x80f40e, 0x0a },
	{ 0x80f40f, 0x40 },
	{ 0x80f410, 0x08 },
	{ 0x80f55f, 0x0a },
	{ 0x80f561, 0x15 },
	{ 0x80f562, 0x20 },
	{ 0x80f5e3, 0x09 },
	{ 0x80f5e4, 0x01 },
	{ 0x80f5e5, 0x01 },
	{ 0x80f600, 0x05 },
	{ 0x80f601, 0x08 },
	{ 0x80f602, 0x0b },
	{ 0x80f603, 0x0e },
	{ 0x80f604, 0x11 },
	{ 0x80f605, 0x14 },
	{ 0x80f606, 0x17 },
	{ 0x80f607, 0x1f },
	{ 0x80f60e, 0x00 },
	{ 0x80f60f, 0x04 },
	{ 0x80f610, 0x32 },
	{ 0x80f611, 0x10 },
	{ 0x80f707, 0xfc },
	{ 0x80f708, 0x00 },
	{ 0x80f709, 0x37 },
	{ 0x80f70a, 0x00 },
	{ 0x80f78b, 0x01 },
	{ 0x80f80f, 0x40 },
	{ 0x80f810, 0x54 },
	{ 0x80f811, 0x5a },
	{ 0x80f905, 0x01 },
	{ 0x80fb06, 0x03 },
	{ 0x80fd8b, 0x00 },
};

/*
 * ITE Tech IT9133 BX Omega LNA config 2 tuner init
 * AF9033_TUNER_IT9135_62   = 0x62
 */
static const struct reg_val tuner_init_it9135_62[] = {
	{ 0x800043, 0x00 },
	{ 0x800046, 0x62 },
	{ 0x800051, 0x01 },
	{ 0x80005f, 0x00 },
	{ 0x800060, 0x00 },
	{ 0x800068, 0x0a },
	{ 0x80006a, 0x03 },
	{ 0x800070, 0x0a },
	{ 0x800071, 0x05 },
	{ 0x800072, 0x02 },
	{ 0x800075, 0x8c },
	{ 0x800076, 0x8c },
	{ 0x800077, 0x8c },
	{ 0x800078, 0x8c },
	{ 0x800079, 0x01 },
	{ 0x80007e, 0x04 },
	{ 0x800081, 0x0a },
	{ 0x800082, 0x12 },
	{ 0x800084, 0x0a },
	{ 0x800085, 0x33 },
	{ 0x800086, 0xb8 },
	{ 0x800087, 0x9c },
	{ 0x800088, 0xb2 },
	{ 0x800089, 0xa6 },
	{ 0x80008a, 0x01 },
	{ 0x80008e, 0x01 },
	{ 0x800092, 0x06 },
	{ 0x800093, 0x00 },
	{ 0x800094, 0x00 },
	{ 0x800095, 0x00 },
	{ 0x800096, 0x00 },
	{ 0x800099, 0x01 },
	{ 0x80009b, 0x3c },
	{ 0x80009c, 0x28 },
	{ 0x80009f, 0xe1 },
	{ 0x8000a0, 0xcf },
	{ 0x8000a3, 0x01 },
	{ 0x8000a4, 0x5a },
	{ 0x8000a5, 0x01 },
	{ 0x8000a6, 0x01 },
	{ 0x8000a9, 0x00 },
	{ 0x8000aa, 0x01 },
	{ 0x8000b0, 0x01 },
	{ 0x8000b3, 0x02 },
	{ 0x8000b4, 0x3a },
	{ 0x8000b6, 0x14 },
	{ 0x8000c0, 0x11 },
	{ 0x8000c1, 0x00 },
	{ 0x8000c2, 0x05 },
	{ 0x8000c3, 0x01 },
	{ 0x8000c4, 0x00 },
	{ 0x8000c6, 0x19 },
	{ 0x8000c7, 0x00 },
	{ 0x8000cb, 0x32 },
	{ 0x8000cc, 0x2c },
	{ 0x8000cd, 0x4f },
	{ 0x8000ce, 0x30 },
	{ 0x8000f3, 0x05 },
	{ 0x8000f4, 0x8c },
	{ 0x8000f5, 0x8c },
	{ 0x8000f8, 0x03 },
	{ 0x8000f9, 0x06 },
	{ 0x8000fa, 0x06 },
	{ 0x8000fc, 0x02 },
	{ 0x8000fd, 0x03 },
	{ 0x8000fe, 0x02 },
	{ 0x8000ff, 0x09 },
	{ 0x800100, 0x50 },
	{ 0x800101, 0x6e },
	{ 0x800102, 0x8c },
	{ 0x800103, 0x02 },
	{ 0x800104, 0x02 },
	{ 0x800105, 0xc2 },
	{ 0x800106, 0x00 },
	{ 0x800109, 0x02 },
	{ 0x800115, 0x0a },
	{ 0x800116, 0x03 },
	{ 0x80011a, 0xb8 },
	{ 0x800124, 0xa8 },
	{ 0x800127, 0x00 },
	{ 0x80012a, 0x53 },
	{ 0x80012b, 0x51 },
	{ 0x80012c, 0x4e },
	{ 0x80012d, 0x43 },
	{ 0x800137, 0x00 },
	{ 0x80013b, 0x05 },
	{ 0x80013f, 0x5b },
	{ 0x800141, 0x59 },
	{ 0x800142, 0xf9 },
	{ 0x800143, 0x59 },
	{ 0x800144, 0x00 },
	{ 0x800145, 0x8c },
	{ 0x800146, 0x8c },
	{ 0x800147, 0x8c },
	{ 0x800148, 0x7b },
	{ 0x800149, 0x8c },
	{ 0x80014a, 0x50 },
	{ 0x80014b, 0x70 },
	{ 0x80014d, 0x96 },
	{ 0x80014e, 0xd0 },
	{ 0x80014f, 0x03 },
	{ 0x800151, 0x28 },
	{ 0x800153, 0xb2 },
	{ 0x800178, 0x09 },
	{ 0x800181, 0x9c },
	{ 0x800182, 0x6e },
	{ 0x800185, 0x24 },
	{ 0x800189, 0xb8 },
	{ 0x80018c, 0x03 },
	{ 0x80018d, 0x5f },
	{ 0x80018f, 0xfb },
	{ 0x800190, 0x5a },
	{ 0x80ed02, 0xff },
	{ 0x80ee42, 0xff },
	{ 0x80ee82, 0xff },
	{ 0x80f000, 0x0f },
	{ 0x80f01f, 0x8c },
	{ 0x80f020, 0x00 },
	{ 0x80f029, 0x8c },
	{ 0x80f02a, 0x00 },
	{ 0x80f02b, 0x00 },
	{ 0x80f064, 0x03 },
	{ 0x80f065, 0xf9 },
	{ 0x80f066, 0x03 },
	{ 0x80f067, 0x01 },
	{ 0x80f06f, 0xe0 },
	{ 0x80f070, 0x03 },
	{ 0x80f072, 0x0f },
	{ 0x80f073, 0x03 },
	{ 0x80f077, 0x01 },
	{ 0x80f078, 0x00 },
	{ 0x80f087, 0x00 },
	{ 0x80f09b, 0x3f },
	{ 0x80f09c, 0x00 },
	{ 0x80f09d, 0x20 },
	{ 0x80f09e, 0x00 },
	{ 0x80f09f, 0x0c },
	{ 0x80f0a0, 0x00 },
	{ 0x80f130, 0x04 },
	{ 0x80f132, 0x04 },
	{ 0x80f144, 0x1a },
	{ 0x80f146, 0x00 },
	{ 0x80f14a, 0x01 },
	{ 0x80f14c, 0x00 },
	{ 0x80f14d, 0x00 },
	{ 0x80f14f, 0x04 },
	{ 0x80f158, 0x7f },
	{ 0x80f15a, 0x00 },
	{ 0x80f15b, 0x08 },
	{ 0x80f15d, 0x03 },
	{ 0x80f15e, 0x05 },
	{ 0x80f163, 0x05 },
	{ 0x80f166, 0x01 },
	{ 0x80f167, 0x40 },
	{ 0x80f168, 0x0f },
	{ 0x80f17a, 0x00 },
	{ 0x80f17b, 0x00 },
	{ 0x80f183, 0x01 },
	{ 0x80f19d, 0x40 },
	{ 0x80f1bc, 0x36 },
	{ 0x80f1bd, 0x00 },
	{ 0x80f1cb, 0xa0 },
	{ 0x80f1cc, 0x01 },
	{ 0x80f204, 0x10 },
	{ 0x80f214, 0x00 },
	{ 0x80f24c, 0x88 },
	{ 0x80f24d, 0x95 },
	{ 0x80f24e, 0x9a },
	{ 0x80f24f, 0x90 },
	{ 0x80f25a, 0x07 },
	{ 0x80f25b, 0xe8 },
	{ 0x80f25c, 0x03 },
	{ 0x80f25d, 0xb0 },
	{ 0x80f25e, 0x04 },
	{ 0x80f270, 0x01 },
	{ 0x80f271, 0x02 },
	{ 0x80f272, 0x01 },
	{ 0x80f273, 0x02 },
	{ 0x80f40e, 0x0a },
	{ 0x80f40f, 0x40 },
	{ 0x80f410, 0x08 },
	{ 0x80f55f, 0x0a },
	{ 0x80f561, 0x15 },
	{ 0x80f562, 0x20 },
	{ 0x80f5e3, 0x09 },
	{ 0x80f5e4, 0x01 },
	{ 0x80f5e5, 0x01 },
	{ 0x80f600, 0x05 },
	{ 0x80f601, 0x08 },
	{ 0x80f602, 0x0b },
	{ 0x80f603, 0x0e },
	{ 0x80f604, 0x11 },
	{ 0x80f605, 0x14 },
	{ 0x80f606, 0x17 },
	{ 0x80f607, 0x1f },
	{ 0x80f60e, 0x00 },
	{ 0x80f60f, 0x04 },
	{ 0x80f610, 0x32 },
	{ 0x80f611, 0x10 },
	{ 0x80f707, 0xfc },
	{ 0x80f708, 0x00 },
	{ 0x80f709, 0x37 },
	{ 0x80f70a, 0x00 },
	{ 0x80f78b, 0x01 },
	{ 0x80f80f, 0x40 },
	{ 0x80f810, 0x54 },
	{ 0x80f811, 0x5a },
	{ 0x80f905, 0x01 },
	{ 0x80fb06, 0x03 },
	{ 0x80fd8b, 0x00 },
};

/* NorDig power reference table */
static const int power_reference[][5] = {
	{-93, -91, -90, -89, -88}, /* QPSK 1/2 ~ 7/8 */
	{-87, -85, -84, -83, -82}, /* 16QAM 1/2 ~ 7/8 */
	{-82, -80, -78, -77, -76}, /* 64QAM 1/2 ~ 7/8 */
};
#endif /* AF9033_PRIV_H */
