/*
     This file is part of GNUnet.
     (C) 2012 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * @file src/gns/gnunet-gns-gtk.c
 * @brief Main function of gnunet-gns-gtk
 * @author Christian Grothoff
 */
#include "gnunet_gtk.h"
#include "gnunet-gns-gtk.h"
#include <gnunet/gnunet_namestore_service.h>


/**
 * Columns in the gns model.
 */
enum GNS_ModelColumns
  {
    /**
     * A gchararray
     */
    GNS_MC_NAME = 0,

    /**
     * A gboolean
     */
    GNS_MC_IS_PUBLIC = 1,

    /**
     * A guint
     */
    GNS_MC_RECORD_TYPE = 2,

    /**
     * A gchararray
     */
    GNS_MC_RECORD_TYPE_AS_STRING = 3,

    /**
     * A guint64
     */
    GNS_MC_EXPIRATION_TIME = 4,

    /**
     * A gboolean
     */
    GNS_MC_EXPIRATION_TIME_IS_RELATIVE = 5,

    /**
     * A gchararray
     */
    GNS_MC_EXPIRATION_TIME_AS_STRING = 6,

    /**
     * A gchararray
     */
    GNS_MC_VALUE_AS_STRING = 7,

    /**
     * A gchararray
     */
    GNS_MC_VALUE_COLOR = 8,
  };


/**
 * Columns in the gns type model.
 */
enum GNS_TypeModelColumns
  {
    /**
     * A guint
     */
    GNS_TYPE_MC_TYPE  = 0,

    /**
     * A gchararray
     */
    GNS_TYPE_MC_TYPENAME = 1,
  };


/**
 * Handle to our main loop.
 */
static struct GNUNET_GTK_MainLoop *ml;

/**
 * Should gnunet-gns-gtk start in tray mode?
 */
static int tray_only;


/**
 * Name of our zone as a string.
 */
static char *zone_as_string;

/**
 * Default directory of zone files as a string.
 */
static char *zonekey_directory;

/**
 * Get cfg.
 */
static const struct GNUNET_CONFIGURATION_Handle *
get_configuration ()
{
  return GNUNET_GTK_main_loop_get_configuration (ml);
}


/**
 * Get an object from the main window.
 *
 * @param name name of the object
 * @return NULL on error
 */
static GObject *
get_object (const char *name)
{
  return GNUNET_GTK_main_loop_get_object (ml, name);
}


void pseu_change_cont (void *cls,
                 int32_t success,
                 const char *emsg)
{
  struct GNUNET_GNS_Context *gns = cls;
  GtkWidget *dialog;
  if (GNUNET_SYSERR == success)
  {
  GNUNET_log (GNUNET_ERROR_TYPE_ERROR, _("New Pseudonym could not be set: `%s'\n"), emsg);
  dialog = gtk_message_dialog_new (GTK_WINDOW (gns->main_window),
                                   GTK_DIALOG_DESTROY_WITH_PARENT,
                                   GTK_MESSAGE_ERROR,
                                   GTK_BUTTONS_CLOSE,
                                   _("New Pseudonym could not be set: `%s'\n"),
                                   emsg);
    g_signal_connect_swapped (dialog, "response",
                              G_CALLBACK (gtk_widget_destroy),
                              dialog);
    gtk_widget_show_all (dialog);
  }
}


/**
 * Task run on shutdown.
 *
 * @param cls unused
 * @param tc scheduler context, unused
 */
static void
shutdown_task (void *cls, const struct GNUNET_SCHEDULER_TaskContext *tc)
{

  struct GNUNET_GNS_Context *gns = cls;
  if (NULL == gns)
    return;
  if (NULL != gns->ns)
  {
    GNUNET_NAMESTORE_disconnect (gns->ns, GNUNET_NO);
    gns->ns = NULL;
  }
  if (NULL != gns->pkey)
  {
    GNUNET_CRYPTO_rsa_key_free (gns->pkey);
    gns->pkey = NULL;
  }
  if (NULL != zonekey_directory)
  {
    GNUNET_free (zonekey_directory);
    zonekey_directory = NULL;
  }
  GNUNET_free (gns);
}


void
GNUNET_GNS_GTK_shutdown (struct GNUNET_GNS_Context *gns)
{
  GNUNET_GTK_tray_icon_destroy ();
  GNUNET_GTK_main_loop_quit (ml);
  GNUNET_SCHEDULER_add_now (&shutdown_task, gns);
}

gboolean
GNUNET_GNS_GTK_pseu_entry_enter_cb (GtkWidget *widget,
                                    GdkEvent  *event,
                                    gpointer   user_data)
{
  const gchar * pseu;

  pseu = gtk_entry_get_text (GTK_ENTRY(widget));

  if ((pseu == NULL) || (0 == strcmp (pseu, "")))
  {
    //gtk_entry_set_text (GTK_ENTRY(widget), PSEU_EMPTY_STR);
  }
  return FALSE;
}


/**
 * The user edited the preferred name (PSEU) of this namespace.
 * Push the update to the namestore.
 *
 * @param editable the edited widget
 * @param user_data unused
 */
void
GNUNET_GNS_GTK_pseu_entry_changed_cb (GtkEditable *editable,
				      gpointer user_data)
{
  struct GNUNET_GNS_Context *gns = user_data;
  struct GNUNET_NAMESTORE_RecordData rd;
  const gchar * pseu;

  pseu = gtk_entry_get_text (GTK_ENTRY(editable));
  if ((pseu != NULL) && (0 != strcmp (pseu, PSEU_EMPTY_STR)) && (0 != strcmp ("", pseu)) && (GNUNET_NO == gns->iteration))
  {

    rd.record_type = GNUNET_NAMESTORE_TYPE_PSEU;
    rd.expiration = GNUNET_TIME_UNIT_FOREVER_ABS;
    rd.flags = GNUNET_NAMESTORE_RF_AUTHORITY;
    rd.data_size = strlen (pseu) + 1;
    rd.data = strdup (pseu);
    GNUNET_NAMESTORE_record_create(gns->ns, gns->pkey, "+", &rd, pseu_change_cont, gns);
    GNUNET_log (GNUNET_ERROR_TYPE_DEBUG, "New Pseudonym is `%s' %u\n", (char *) rd.data, rd.data_size);
  }
  else if ((0 != strcmp (pseu, PSEU_EMPTY_STR)) && ((pseu == NULL) || (0 == strcmp ("", pseu))))
  {
    gtk_entry_set_text (GTK_ENTRY(editable), PSEU_EMPTY_STR);
  }

}

/**
 * The user toggled the 'autoshort' option.  Update the configuration.
 *
 * @param checkmenuitem the menu item
 * @param user_data unused
 */
void
GNUNET_GNS_GTK_autoshort_imagemenuitem_toggled_cb (GtkCheckMenuItem *checkmenuitem,
						   gpointer user_data)
{
  struct GNUNET_GNS_Context *gns = user_data;
  GtkWidget *dialog;
  struct GNUNET_CONFIGURATION_Handle *cfg = (struct GNUNET_CONFIGURATION_Handle *) get_configuration();

  gboolean state = gtk_check_menu_item_get_active (gns->shorten_menu);
  if (TRUE == state)
    GNUNET_CONFIGURATION_set_value_string(cfg,"gns", "AUTO_IMPORT_PKEY","YES");
  else
    GNUNET_CONFIGURATION_set_value_string(cfg,"gns", "AUTO_IMPORT_PKEY","NO");

  char * cfgfile = strdup (GNUNET_GTK_main_loop_get_configuration_file(ml));
  if (GNUNET_SYSERR == GNUNET_CONFIGURATION_write(cfg, cfgfile))
  {
    GNUNET_log (GNUNET_ERROR_TYPE_ERROR, _("Changes to autoshorten could not be written to configuration file: `%s'\n"), cfgfile);
    dialog = gtk_message_dialog_new (GTK_WINDOW (gns->main_window),
                                     GTK_DIALOG_DESTROY_WITH_PARENT,
                                     GTK_MESSAGE_ERROR,
                                     GTK_BUTTONS_CLOSE,
                                     _("Changes to autoshorten option could not be written to configuration file: `%s'\n"),
                                     cfgfile);
    g_signal_connect_swapped (dialog, "response",
                              G_CALLBACK (gtk_widget_destroy),
                              dialog);
    gtk_widget_show_all (dialog);
  }
  GNUNET_free (cfgfile);
}


/**
 * The user selected 'NEW' in the menu.  Open a dialog to enter a filename
 * to create a new zone (for editing).
 *
 * @param checkmenuitem the menu item
 * @param user_data unused
 */
void
GNUNET_GNS_GTK_new_imagemenuitem_activate_cb (GtkMenuItem *menuitem,
					      gpointer user_data)
{
  GNUNET_break (0); // FIXME, not implemented
}


/**
 * Function called from the open-directory dialog upon completion.
 *
 * @param dialog the pseudonym selection dialog
 * @param response_id response code from the dialog
 * @param user_data the builder of the dialog
 */
void
GNUNET_GNS_GTK_zone_open_dialog_response_cb (GtkDialog * dialog,
					     gint response_id,
					     gpointer user_data)
{
  char *filename;

  if (GTK_RESPONSE_OK != response_id)
  {
    gtk_widget_destroy (GTK_WIDGET (dialog));
    g_object_unref (G_OBJECT (dialog));
    return;
  }
  filename = GNUNET_GTK_filechooser_get_filename_utf8 (GTK_FILE_CHOOSER (dialog));
  gtk_widget_destroy (GTK_WIDGET (dialog));
  g_object_unref (G_OBJECT (dialog));

  /* FIXME: move to new zone 'filename' */
  fprintf (stderr, "Got zone `%s'\n", filename);
  GNUNET_free (filename);  
}


/**
 * The user selected 'OPEN' in the menu.  Open a dialog to select
 * a different zonefile (for editing).
 *
 * @param checkmenuitem the menu item
 * @param user_data unused
 */
void
GNUNET_GNS_GTK_open_imagemenuitem_activate_cb (GtkMenuItem *menuitem,
					       gpointer user_data)
{
  GtkWidget *ad;
  GtkBuilder *builder;
  GtkWidget *toplevel;
  GtkFileFilter *ff;
  GtkFileChooser *fc;

  builder =
    GNUNET_GTK_get_new_builder ("gnunet_gns_gtk_zone_open.glade", NULL);
  if (NULL == builder)
  {
    GNUNET_break (0);
    return;
  }
  /* This file filter could be set with glade if we use gtk3
   * With gtk2 we have to set it manually */
  ff = GTK_FILE_FILTER (gtk_builder_get_object
      (builder, "GNUNET_GNS_GTK_zone_open_filefilter"));
  gtk_file_filter_add_pattern (ff, "*.zkey");

  ad = GTK_WIDGET (gtk_builder_get_object
                   (builder, "GNUNET_GNS_GTK_zone_open_filechooserdialog"));

  if (GTK_IS_FILE_CHOOSER(ad))
  {
    fc = GTK_FILE_CHOOSER(ad);
    if (NULL != fc)
      gtk_file_chooser_set_current_folder(fc, zonekey_directory);
  }

  toplevel = gtk_widget_get_toplevel (GTK_WIDGET (menuitem));
  if (GTK_IS_WINDOW (toplevel))
    gtk_window_set_transient_for (GTK_WINDOW (ad), GTK_WINDOW (toplevel));
  gtk_window_present (GTK_WINDOW (ad));
}


/**
 * The user clicked on the 'copy' button.  Copy the full string
 * with the hash of our public key to the clipboard.
 *
 * @param button the button that was clicked
 * @param user_data unused
 */
void
GNUNET_GNS_GTK_public_key_copy_button_clicked_cb (GtkButton *button,
						  gpointer user_data)
{
  GtkClipboard *cb;

  cb = gtk_clipboard_get (GDK_SELECTION_CLIPBOARD);
  gtk_clipboard_set_text (cb, zone_as_string, -1);
}



/**
 * Callback invoked if the application is supposed to exit (via menu).
 *
 * @param menuitem the quit menu
 * @param user_data unused
 */
void
GNUNET_GNS_GTK_quit_imagemenuitem_activate_cb (GtkMenuItem *menuitem,
					       gpointer user_data)
{
  GNUNET_GTK_tray_icon_destroy ();
  GNUNET_GTK_main_loop_quit (ml);
  GNUNET_SCHEDULER_add_now (&shutdown_task, user_data);
}


/**
 * Callback invoked if the application is supposed to exit (via window-close).
 *
 * @param widget the main window
 * @param event deletion event
 * @param user_data unused
 */
void
GNUNET_GNS_GTK_main_window_delete_event_cb (GtkWidget *widget,
					    GdkEvent *event,
					    gpointer user_data)
{
  GNUNET_GTK_tray_icon_destroy ();
  GNUNET_GTK_main_loop_quit (ml);
  GNUNET_SCHEDULER_add_now (&shutdown_task, user_data);
}


void
close_error_box (GtkDialog *dialog,
                gint       response_id,
                gpointer   user_data)
{
  gtk_widget_destroy (GTK_WIDGET(dialog));
  GNUNET_GNS_GTK_shutdown (user_data);
}

static void
namestore_service_check (void *cls, const struct GNUNET_SCHEDULER_TaskContext *tc)
{
  struct GNUNET_GNS_Context *gns = NULL;
  struct GNUNET_CRYPTO_ShortHashAsciiEncoded shenc;
  GtkWidget *dialog;
  char *label;
  char *keyfile;
  char *servicehome;

  gns = GNUNET_malloc (sizeof (struct GNUNET_GNS_Context));

  if ((tc->reason & GNUNET_SCHEDULER_REASON_TIMEOUT) != 0)
  {
   char * message = _("Namestore service is not running!\n");
   GNUNET_log (GNUNET_ERROR_TYPE_ERROR, message);
   dialog = gtk_message_dialog_new (GTK_WINDOW (gns->main_window),
                                    GTK_DIALOG_DESTROY_WITH_PARENT,
                                    GTK_MESSAGE_ERROR,
                                    GTK_BUTTONS_CLOSE,
                                    "%s",
                                    message);

   g_signal_connect (dialog, "response", G_CALLBACK(close_error_box), gns);
   gtk_widget_show_all (dialog);
   return;
  }

  if (GNUNET_OK != GNUNET_CONFIGURATION_get_value_filename (get_configuration (),
                                                            "PATHS",
                                                            "SERVICEHOME",
                                                            &servicehome))
  {
    GNUNET_asprintf(&zonekey_directory, "");
  }
  else
  {
    GNUNET_asprintf(&zonekey_directory, "%s%s%s",servicehome, DIR_SEPARATOR_STR, "gns");
    GNUNET_free (servicehome);
  }

  /* setup crypto keys */
  if (GNUNET_OK != GNUNET_CONFIGURATION_get_value_filename (get_configuration (),
                                                            "gns",
                                                            "ZONEKEY",
                                                            &keyfile))
  {
    GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                _("Option `%s' missing in section `%s'\n"), "ZONEKEY", "gns");
    return;
  }
  else
    GNUNET_log (GNUNET_ERROR_TYPE_DEBUG, "Using `%s'\n", keyfile);
  gns->pkey = GNUNET_CRYPTO_rsa_key_create_from_file (keyfile);
  GNUNET_free (keyfile);
  keyfile = NULL;
  if (NULL == gns->pkey)
  {
    GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                _("Failed to read or create private zone key\n"));
    return;
  }
  GNUNET_CRYPTO_rsa_key_get_public (gns->pkey, &gns->pubkey);
  GNUNET_CRYPTO_short_hash (&gns->pubkey,
                            sizeof (struct GNUNET_CRYPTO_RsaPublicKeyBinaryEncoded),
                            &gns->zone);
  GNUNET_CRYPTO_short_hash_to_enc(&gns->zone, &shenc);

  /* connect to namestore */
  gns->ns = GNUNET_NAMESTORE_connect (get_configuration ());
  if (NULL == gns->ns)
  {
    GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                _("Failed to connect to namestore\n"));
    GNUNET_SCHEDULER_add_now (&shutdown_task, gns);
    return;
  }

  /* setup gui */
  if (GNUNET_OK != GNUNET_GTK_main_loop_build_window (ml, gns))
  {
    GNUNET_break (0);
    GNUNET_SCHEDULER_add_now (&shutdown_task, gns);
    return;
  }
  gns->builder = GNUNET_GTK_main_loop_get_builder(ml);
  gns->main_window = GTK_WIDGET (get_object ("GNUNET_GNS_GTK_main_window"));
  gns->ts  = GTK_TREE_STORE (gtk_builder_get_object (gns->builder, "GNUNET_GNS_GTK_treestore"));
  gns->ls =  GTK_LIST_STORE (gtk_builder_get_object (gns->builder, "GNUNET_GNS_GTK_type_liststore"));
  gns->tv =  GTK_TREE_VIEW (gtk_builder_get_object (gns->builder, "GNUNET_GNS_GTK_main_treeview"));
  gns->tm = GTK_TREE_MODEL(gns->ts);
  gns->shorten_menu =  GTK_CHECK_MENU_ITEM(gtk_builder_get_object (gns->builder, "GNUNET_GNS_GTK_autoshort_imagemenuitem"));
  if (GNUNET_YES == GNUNET_CONFIGURATION_get_value_yesno (get_configuration (),
                                                            "gns",
                                                            "AUTO_IMPORT_PKEY"))
    gtk_check_menu_item_set_active (gns->shorten_menu, TRUE);
  else
    gtk_check_menu_item_set_active (gns->shorten_menu, FALSE);

  /* TODO: implements menus */
  gtk_widget_set_visible (GTK_WIDGET (gtk_builder_get_object (gns->builder, "GNUNET_GNS_GTK_new_imagemenuitem")), FALSE);
  gtk_widget_set_visible (GTK_WIDGET (gtk_builder_get_object (gns->builder, "GNUNET_GNS_GTK_open_imagemenuitem")), FALSE);

  zone_as_string = GNUNET_strdup ((char *) &shenc);
  label = g_markup_printf_escaped (_("<b>Editing zone %s</b>"),
                                  zone_as_string);
  gtk_label_set_markup (GTK_LABEL (get_object ("GNUNET_GNS_GTK_zone_label")),
                       label);
  g_free (label);

  GNUNET_GTK_set_icon_search_path ();
  GNUNET_GTK_setup_nls ();
  /* setup main window */
  GNUNET_GTK_tray_icon_create (GTK_WINDOW (gns->main_window),
                              "gnunet-gtk" /* FIXME: different icon? */ ,
                              "gnunet-gns-gtk");

  /* make GUI visible */
  if (!tray_only)
  {
   gtk_widget_show (gns->main_window);
   gtk_window_present (GTK_WINDOW (gns->main_window));
  }
}


/**
 * Actual main function run right after GNUnet's scheduler
 * is initialized.  Initializes up GTK and Glade.
 *
 * @param cls the 'struct GNUNET_GTK_MainLoop'
 * @param tc scheduler context
 */
static void
run (void *cls, const struct GNUNET_SCHEDULER_TaskContext *tc)
{
  ml = cls;
  GNUNET_CLIENT_service_test("namestore", get_configuration(), GNUNET_TIME_UNIT_SECONDS, &namestore_service_check, NULL);
}


int
main (int argc, char *const *argv)
{
  static struct GNUNET_GETOPT_CommandLineOption options[] = {
    {'t', "tray", NULL,
     gettext_noop ("start in tray mode"), 0,
     &GNUNET_GETOPT_set_one, &tray_only},

    GNUNET_GETOPT_OPTION_END
  };

  if (GNUNET_OK !=
      GNUNET_GTK_main_loop_start ("gnunet-gns-gtk",
                                  "GTK GUI for editing our zone", argc,
                                  argv, options,
                                  "gnunet_gns_gtk_main_window.glade",
                                  &run))
    return 1;
  return 0;
}


/* end of gnunet-gns-gtk.c */
